<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use App\Traits\HasTenantScope;
use App\Mail\ApplicationStatusChanged;

class Application extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'job_posting_id',
        'candidate_id',
        'first_name',
        'last_name',
        'email',
        'phone',
        'cv_path',
        'cover_letter',
        'status',
        'screening_score',
        'notes',
        'tracking_token',
        'applied_at',
        'organization_id',
        'updated_count',
        'last_updated_at',
    ];

    protected $casts = [
        'applied_at' => 'datetime',
        'last_updated_at' => 'datetime',
        'screening_score' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($application) {
            if (empty($application->tracking_token)) {
                $application->tracking_token = Str::random(32);
            }
            if (empty($application->applied_at)) {
                $application->applied_at = now();
            }
        });

        static::created(function ($application) {
            // Create status history record
            ApplicationStatusHistory::create([
                'application_id' => $application->id,
                'from_status' => null,
                'to_status' => $application->status,
                'changed_by' => auth()->id() ?? 1, // System user if no auth
                'notes' => 'Application submitted',
            ]);
        });

        static::updated(function ($application) {
            // Track status changes
            if ($application->isDirty('status')) {
                $oldStatus = $application->getOriginal('status');
                $newStatus = $application->status;
                
                // Create status history record
                ApplicationStatusHistory::create([
                    'application_id' => $application->id,
                    'from_status' => $oldStatus,
                    'to_status' => $newStatus,
                    'changed_by' => auth()->id() ?? 1,
                    'notes' => 'Status updated',
                ]);

                // Send status change email using organization's email settings
                try {
                    // Configure organization email settings
                    if ($application->organization && $application->organization->use_custom_mail) {
                        \App\Services\OrganizationMailService::configureForOrganization($application->organization);
                    }
                    
                    Mail::to($application->email)->send(
                        new ApplicationStatusChanged($application, $oldStatus, $newStatus, auth()->user())
                    );
                    \Log::info('Status change email sent', [
                        'application_id' => $application->id,
                        'from_status' => $oldStatus,
                        'to_status' => $newStatus
                    ]);
                } catch (\Exception $e) {
                    \Log::error('Failed to send status change email: ' . $e->getMessage(), [
                        'application_id' => $application->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }
        });
    }

    /**
     * Get the job posting that owns the application
     */
    public function jobPosting()
    {
        return $this->belongsTo(JobPosting::class);
    }

    /**
     * Get the candidate that owns the application
     */
    public function candidate()
    {
        return $this->belongsTo(Candidate::class);
    }

    /**
     * Get the organization that owns the application
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get all interviews for this application
     */
    public function interviews()
    {
        return $this->hasMany(Interview::class);
    }

    /**
     * Get all status history for this application
     */
    public function statusHistory()
    {
        return $this->hasMany(ApplicationStatusHistory::class)->orderBy('created_at');
    }

    /**
     * Get all messages for this application
     */
    public function messages()
    {
        return $this->hasMany(ApplicationMessage::class)->orderBy('created_at');
    }

    /**
     * Get the job offer for this application
     */
    public function jobOffer()
    {
        return $this->hasOne(JobOffer::class);
    }

    /**
     * Get the employee record if converted
     */
    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * Get the full name attribute
     */
    public function getFullNameAttribute()
    {
        return "{$this->first_name} {$this->last_name}";
    }

    /**
     * Get the CV download URL
     */
    public function getCvUrlAttribute()
    {
        return $this->cv_path ? route('applications.cv.view', $this) : null;
    }

    /**
     * Get the CV download URL
     */
    public function getCvDownloadUrlAttribute()
    {
        return $this->cv_path ? route('applications.cv.download', $this) : null;
    }

    /**
     * Get the tracking URL
     */
    public function getTrackingUrlAttribute()
    {
        return url("/track/{$this->tracking_token}");
    }

    /**
     * Check if application can be updated
     */
    public function canBeUpdated()
    {
        return $this->status === 'new' && $this->updated_count < 1;
    }

    /**
     * Check if application has reached update limit
     */
    public function hasReachedUpdateLimit()
    {
        return $this->updated_count >= 1;
    }

    /**
     * Check if application can be edited
     */
    public function canBeEdited()
    {
        return in_array($this->status, ['new', 'screening']);
    }

    /**
     * Check if application can be withdrawn
     */
    public function canBeWithdrawn()
    {
        return in_array($this->status, ['new', 'screening']);
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'new' => 'blue',
            'screening' => 'yellow',
            'shortlisted' => 'purple',
            'interviewed' => 'indigo',
            'offered' => 'green',
            'rejected' => 'red',
            'withdrawn' => 'gray',
            default => 'gray',
        };
    }

    /**
     * Get next steps message based on status
     */
    public function getNextStepsAttribute()
    {
        return match($this->status) {
            'new' => 'Your application is being reviewed by our team.',
            'screening' => 'Your application is currently under detailed review.',
            'shortlisted' => 'Congratulations! You have been shortlisted. We will contact you soon to schedule an interview.',
            'interviewed' => 'Thank you for interviewing with us. We are currently making our final decision.',
            'offered' => 'Congratulations! We have extended an offer to you. Please check your email for details.',
            'rejected' => 'Thank you for your interest. Unfortunately, we have decided to move forward with other candidates.',
            'withdrawn' => 'Your application has been withdrawn.',
            default => 'Your application is being processed.',
        };
    }

    /**
     * Check if model belongs to current organization
     */
    public function belongsToCurrentOrganization(): bool
    {
        $currentOrgId = static::getCurrentOrganizationId();
        return $currentOrgId && $this->organization_id == $currentOrgId;
    }

    /**
     * Scope to filter by status
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope to filter by job posting
     */
    public function scopeForJobPosting($query, $jobPostingId)
    {
        return $query->where('job_posting_id', $jobPostingId);
    }
}