<?php

namespace App\Mail;

use App\Models\Application;
use App\Models\JobPosting;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;
use App\Mail\Concerns\UsesOrganizationMailSettings;

class ApplicationStatusChanged extends Mailable implements ShouldQueue
{
    use Queueable, SerializesModels, UsesOrganizationMailSettings;

    public $application;
    public $jobPosting;
    public $oldStatus;
    public $newStatus;
    public $changedBy;
    public $notes;

    /**
     * Create a new message instance.
     */
    public function __construct(Application $application, $oldStatus, $newStatus, $changedBy = null, $notes = null)
    {
        $this->application = $application;
        $this->jobPosting = $application->jobPosting;
        $this->oldStatus = $oldStatus;
        $this->newStatus = $newStatus;
        $this->changedBy = $changedBy;
        $this->notes = $notes;
    }

    /**
     * Get the organization for mail configuration
     */
    protected function getOrganization()
    {
        return $this->jobPosting->organization ?? null;
    }

    /**
     * Get the message envelope.
     */
    public function envelope(): Envelope
    {
        $subject = "Application Update - {$this->jobPosting->title}";
        
        $fromAddress = $this->jobPosting->organization->use_custom_mail && $this->jobPosting->organization->mail_from_address
            ? $this->jobPosting->organization->mail_from_address
            : config('mail.from.address', 'noreply@hrms.com');
            
        $fromName = $this->jobPosting->organization->use_custom_mail && $this->jobPosting->organization->mail_from_name
            ? $this->jobPosting->organization->mail_from_name
            : $this->jobPosting->organization->name;
        
        return new Envelope(
            subject: $subject,
            from: $fromAddress,
            replyTo: $fromAddress,
        );
    }

    /**
     * Get the message content definition.
     */
    public function content(): Content
    {
        return new Content(
            view: 'emails.application-status-changed',
            with: [
                'application' => $this->application,
                'jobPosting' => $this->jobPosting,
                'organizationName' => $this->jobPosting->organization->name,
                'oldStatus' => $this->oldStatus,
                'newStatus' => $this->newStatus,
                'changedBy' => $this->changedBy,
                'notes' => $this->notes,
                'trackingUrl' => $this->application->tracking_url,
                'statusMessage' => $this->getStatusMessage(),
            ]
        );
    }

    /**
     * Get user-friendly status message
     */
    private function getStatusMessage(): string
    {
        switch ($this->newStatus) {
            case 'screening':
                return 'Your application is being reviewed by our HR team.';
            case 'shortlisted':
                return 'Congratulations! You have been shortlisted for this position.';
            case 'interviewed':
                return 'Thank you for your interview. We are reviewing all candidates.';
            case 'offered':
                return 'Congratulations! We would like to offer you this position.';
            case 'rejected':
                return 'Thank you for your interest. We have decided to move forward with other candidates.';
            default:
                return 'Your application status has been updated.';
        }
    }

    /**
     * Get the attachments for the message.
     *
     * @return array<int, \Illuminate\Mail\Mailables\Attachment>
     */
    public function attachments(): array
    {
        return [];
    }
}