<?php

namespace App\Listeners;

use App\Services\AuditService;
use Illuminate\Auth\Events\Login;
use Illuminate\Auth\Events\Logout;
use Illuminate\Auth\Events\Failed;
use Illuminate\Auth\Events\Registered;
use Illuminate\Auth\Events\Verified;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Events\Dispatcher;

class AuditAuthenticationEvents
{
    /**
     * Handle user login events
     */
    public function handleLogin(Login $event): void
    {
        AuditService::logAuth(
            'login',
            "User {$event->user->name} logged in successfully",
            [
                'user_id' => $event->user->id,
                'guard' => $event->guard,
                'remember' => $event->remember,
            ]
        );
    }

    /**
     * Handle user logout events
     */
    public function handleLogout(Logout $event): void
    {
        if ($event->user) {
            AuditService::logAuth(
                'logout',
                "User {$event->user->name} logged out",
                [
                    'user_id' => $event->user->id,
                    'guard' => $event->guard,
                ]
            );
        }
    }

    /**
     * Handle failed login attempts
     */
    public function handleFailed(Failed $event): void
    {
        $email = $event->credentials['email'] ?? 'unknown';
        
        AuditService::logAuth(
            'login_failed',
            "Failed login attempt for email: {$email}",
            [
                'email' => $email,
                'guard' => $event->guard,
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]
        );

        // Check for multiple failed attempts from same IP
        $this->checkForSuspiciousActivity($email);
    }

    /**
     * Handle user registration events
     */
    public function handleRegistered(Registered $event): void
    {
        AuditService::logAuth(
            'user_registered',
            "New user registered: {$event->user->name} ({$event->user->email})",
            [
                'user_id' => $event->user->id,
                'email' => $event->user->email,
            ]
        );
    }

    /**
     * Handle email verification events
     */
    public function handleVerified(Verified $event): void
    {
        AuditService::logAuth(
            'email_verified',
            "User {$event->user->name} verified their email address",
            [
                'user_id' => $event->user->id,
                'email' => $event->user->email,
            ]
        );
    }

    /**
     * Handle password reset events
     */
    public function handlePasswordReset(PasswordReset $event): void
    {
        AuditService::logAuth(
            'password_reset',
            "User {$event->user->name} reset their password",
            [
                'user_id' => $event->user->id,
                'email' => $event->user->email,
            ]
        );
    }

    /**
     * Check for suspicious login activity
     */
    private function checkForSuspiciousActivity(string $email): void
    {
        $recentFailures = \App\Models\AuditLog::where('event_type', 'login_failed')
            ->where('created_at', '>=', now()->subMinutes(15))
            ->where('ip_address', request()->ip())
            ->count();

        if ($recentFailures >= 5) {
            AuditService::logSecurity(
                'multiple_failed_logins',
                "Multiple failed login attempts detected from IP: " . request()->ip(),
                [
                    'ip_address' => request()->ip(),
                    'failed_attempts' => $recentFailures,
                    'target_email' => $email,
                    'time_window' => '15 minutes',
                ]
            );
        }
    }

    /**
     * Register the listeners for the subscriber
     */
    public function subscribe(Dispatcher $events): array
    {
        return [
            Login::class => 'handleLogin',
            Logout::class => 'handleLogout',
            Failed::class => 'handleFailed',
            Registered::class => 'handleRegistered',
            Verified::class => 'handleVerified',
            PasswordReset::class => 'handlePasswordReset',
        ];
    }
}