<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Services\LogoUploadService;
use App\Models\ThemeSetting;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class ProcessLogoUpload implements ShouldQueue
{
    use Queueable, InteractsWithQueue, SerializesModels;

    /**
     * The number of times the job may be attempted.
     */
    public int $tries = 3;

    /**
     * The number of seconds the job can run before timing out.
     */
    public int $timeout = 120;

    /**
     * Create a new job instance.
     */
    public function __construct(
        public string $tempPath,
        public string $logoType,
        public int $organizationId,
        public string $originalExtension
    ) {}

    /**
     * Execute the job.
     */
    public function handle(LogoUploadService $logoService): void
    {
        try {
            // Check if temp file exists
            if (!Storage::disk('local')->exists($this->tempPath)) {
                Log::error("Temp file not found for logo processing: {$this->tempPath}");
                return;
            }

            // Get the temp file
            $tempFile = Storage::disk('local')->path($this->tempPath);
            
            // Create UploadedFile instance
            $uploadedFile = new \Illuminate\Http\UploadedFile(
                $tempFile,
                basename($this->tempPath),
                mime_content_type($tempFile),
                null,
                true
            );

            // Process and upload the logo
            $finalUrl = $logoService->upload(
                $uploadedFile,
                $this->logoType,
                $this->organizationId
            );

            // Update theme setting with final URL
            $themeSetting = ThemeSetting::firstOrCreate(
                ['organization_id' => $this->organizationId],
                ['base_theme' => 'default', 'is_active' => true]
            );

            $themeSetting->update(["{$this->logoType}_url" => $finalUrl]);

            // Clear theme cache
            Cache::forget("theme_settings.org.{$this->organizationId}");

            // Clean up temp file
            Storage::disk('local')->delete($this->tempPath);

            Log::info("Logo processed successfully: {$this->logoType} for org {$this->organizationId}");

        } catch (\Exception $e) {
            Log::error("Failed to process logo upload: " . $e->getMessage(), [
                'logo_type' => $this->logoType,
                'organization_id' => $this->organizationId,
                'error' => $e->getMessage(),
            ]);

            // Clean up temp file on failure
            if (Storage::disk('local')->exists($this->tempPath)) {
                Storage::disk('local')->delete($this->tempPath);
            }

            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error("Logo upload job failed permanently", [
            'logo_type' => $this->logoType,
            'organization_id' => $this->organizationId,
            'error' => $exception->getMessage(),
        ]);

        // Clean up temp file
        if (Storage::disk('local')->exists($this->tempPath)) {
            Storage::disk('local')->delete($this->tempPath);
        }
    }
}

