<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class SetTenantContext
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Skip tenant context for system admin routes or if no user is authenticated
        if (!Auth::check()) {
            return $next($request);
        }

        $user = Auth::user();

        // System admins can access all organizations
        if ($user->isSystemAdmin()) {
            // For system admins, check if they're accessing a specific organization
            $organizationId = $this->getOrganizationFromRequest($request);
            if ($organizationId) {
                app()->instance('tenant.organization_id', $organizationId);
            }
            return $next($request);
        }

        // Regular users are scoped to their organization
        if ($user->organization_id) {
            app()->instance('tenant.organization_id', $user->organization_id);
        }

        return $next($request);
    }

    /**
     * Extract organization ID from request (for system admin context switching)
     */
    private function getOrganizationFromRequest(Request $request): ?int
    {
        // Check for organization parameter in route
        if ($request->route('organization')) {
            $organization = $request->route('organization');
            
            // If it's a model instance (route model binding), get the ID
            if (is_object($organization) && method_exists($organization, 'getKey')) {
                return (int) $organization->getKey();
            }
            
            // If it's already an ID, cast to int
            if (is_numeric($organization)) {
                return (int) $organization;
            }
        }

        // Check for organization_id in query parameters
        if ($request->has('organization_id')) {
            return (int) $request->get('organization_id');
        }

        // Check for organization context in session (for system admin)
        if (session()->has('current_organization_id')) {
            return (int) session('current_organization_id');
        }

        return null;
    }
}