<?php

namespace App\Http\Middleware;

use App\Services\InputSanitizationService;
use App\Services\SecurityMonitoringService;
use Closure;
use Illuminate\Http\Request;

class SanitizeInput
{
    protected $inputSanitizer;
    protected $securityMonitor;

    public function __construct(
        InputSanitizationService $inputSanitizer,
        SecurityMonitoringService $securityMonitor
    ) {
        $this->inputSanitizer = $inputSanitizer;
        $this->securityMonitor = $securityMonitor;
    }

    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next)
    {
        // Monitor suspicious input patterns
        $this->securityMonitor->monitorSuspiciousInput($request);

        // Define sanitization rules for different input types
        $sanitizationRules = [
            'email' => ['type' => 'email'],
            'phone' => ['type' => 'phone'],
            'website' => ['type' => 'url'],
            'description' => [
                'type' => 'html',
                'allowed_tags' => '<p><br><strong><em><ul><ol><li>'
            ],
            'name' => [
                'type' => 'string',
                'options' => [
                    'strip_tags' => true,
                    'max_length' => 255,
                    'remove_special_chars' => false
                ]
            ],
            'search' => [
                'type' => 'string',
                'options' => [
                    'strip_tags' => true,
                    'max_length' => 100
                ]
            ],
        ];

        // Sanitize input data
        $sanitizedData = $this->sanitizeRequestData($request->all(), $sanitizationRules);
        
        // Replace request data with sanitized data
        $request->replace($sanitizedData);

        return $next($request);
    }

    /**
     * Sanitize request data based on field names and rules
     */
    private function sanitizeRequestData(array $data, array $rules): array
    {
        $sanitized = [];

        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $sanitized[$key] = $this->sanitizeRequestData($value, $rules);
            } else {
                // Determine sanitization rule based on field name
                $rule = $this->getSanitizationRule($key, $rules);
                $sanitized[$key] = $this->sanitizeValue($value, $rule);
            }
        }

        return $sanitized;
    }

    /**
     * Get sanitization rule for a field
     */
    private function getSanitizationRule(string $fieldName, array $rules): array
    {
        // Check for exact match
        if (isset($rules[$fieldName])) {
            return $rules[$fieldName];
        }

        // Check for pattern matches
        foreach ($rules as $pattern => $rule) {
            if (str_contains($fieldName, $pattern)) {
                return $rule;
            }
        }

        // Default rule for strings
        return [
            'type' => 'string',
            'options' => [
                'strip_tags' => true,
                'max_length' => 1000
            ]
        ];
    }

    /**
     * Sanitize a single value
     */
    private function sanitizeValue($value, array $rule)
    {
        if (!is_string($value)) {
            return $value;
        }

        $type = $rule['type'] ?? 'string';

        switch ($type) {
            case 'email':
                return $this->inputSanitizer->sanitizeEmail($value);
            
            case 'phone':
                return $this->inputSanitizer->sanitizePhone($value);
            
            case 'url':
                return $this->inputSanitizer->sanitizeUrl($value);
            
            case 'html':
                $allowedTags = $rule['allowed_tags'] ?? '';
                return strip_tags($value, $allowedTags);
            
            case 'string':
            default:
                $options = $rule['options'] ?? [];
                return $this->inputSanitizer->sanitizeString($value, $options);
        }
    }
}