<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckSubscription
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = $request->user();

        // System admins bypass subscription checks
        if ($user && $user->is_system_admin) {
            return $next($request);
        }

        // Check if user has organization
        if (!$user || !$user->organization_id) {
            return redirect()->route('login')->with('error', 'No organization found');
        }

        $organization = $user->organization;
        $subscription = $organization->subscription;

        // No subscription found
        if (!$subscription) {
            return redirect()->route('subscription.required')
                ->with('error', 'Your organization does not have an active subscription');
        }

        // Check subscription status
        if ($subscription->status === 'expired') {
            return redirect()->route('subscription.expired')
                ->with('error', 'Your subscription has expired. Please renew to continue.');
        }

        if ($subscription->status === 'suspended') {
            return redirect()->route('subscription.suspended')
                ->with('error', 'Your subscription has been suspended. Please contact support.');
        }

        if ($subscription->status === 'cancelled') {
            return redirect()->route('subscription.cancelled')
                ->with('error', 'Your subscription has been cancelled.');
        }

        // Check if trial has expired
        if ($subscription->status === 'trial' && $subscription->trialExpired()) {
            return redirect()->route('subscription.trial-expired')
                ->with('error', 'Your trial period has ended. Please subscribe to continue.');
        }

        // All checks passed
        return $next($request);
    }
}
