<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Payroll;
use App\Models\Employee;
use App\Services\PayrollService;
use App\Services\PayrollApprovalService;
use App\Services\KenyaTaxCalculator;
use Illuminate\Http\Request;
use Carbon\Carbon;

class PayrollController extends Controller
{
    protected PayrollService $payrollService;
    protected PayrollApprovalService $approvalService;

    public function __construct(PayrollService $payrollService, PayrollApprovalService $approvalService)
    {
        $this->payrollService = $payrollService;
        $this->approvalService = $approvalService;
    }

    /**
     * Display payroll listing page
     */
    public function index(Request $request)
    {
        $query = Payroll::with(['employee', 'contract']);

        // Filter by year and month
        if ($request->has('year')) {
            $query->where('year', $request->year);
        }
        if ($request->has('month')) {
            $query->where('month', $request->month);
        }

        // Filter by employment type
        if ($request->has('employment_type')) {
            $query->where('employment_type', $request->employment_type);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $payrolls = $query->orderBy('created_at', 'desc')->paginate(20);

        // Get filter options
        $years = range(date('Y') - 2, date('Y') + 1);
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];
        $employmentTypes = ['permanent', 'contract', 'freelance', 'casual', 'intern', 'part_time'];
        $statuses = ['draft', 'approved', 'processed', 'paid'];

        return view('payroll.index', compact('payrolls', 'years', 'months', 'employmentTypes', 'statuses'));
    }

    /**
     * Show bulk payroll generation page
     */
    public function bulk()
    {
        $employees = Employee::where('status', 'active')
            ->whereHas('activeContract')
            ->with(['activeContract', 'department', 'position'])
            ->get();

        $years = range(date('Y') - 1, date('Y') + 1);
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];

        return view('payroll.bulk', compact('employees', 'years', 'months'));
    }

    /**
     * Process bulk payroll generation
     */
    public function bulkStore(Request $request)
    {
        $request->validate([
            'year' => 'required|integer|min:2020|max:2030',
            'month' => 'required|integer|min:1|max:12',
            'employee_ids' => 'sometimes|array',
            'employee_ids.*' => 'exists:employees,id',
        ]);

        try {
            $payrolls = $this->payrollService->generateMonthlyPayroll(
                $request->year,
                $request->month,
                $request->employee_ids ?? []
            );

            return redirect()->route('payroll.index')
                ->with('success', "Payroll generated successfully for {$payrolls->count()} employees.");
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to generate payroll: ' . $e->getMessage()]);
        }
    }

    /**
     * Show create payroll page
     */
    public function create()
    {
        $employees = Employee::where('status', 'active')
            ->whereHas('activeContract')
            ->with(['activeContract', 'department', 'position'])
            ->get();

        // Map employee data for JavaScript
        $employeesForJs = $employees->map(function($employee) {
            return [
                'id' => $employee->id,
                'name' => $employee->full_name,
                'code' => $employee->employee_code,
                'department' => $employee->department->name ?? 'No Department',
                'employment_type' => $employee->activeContract->employment_type ?? 'N/A',
                'payment_frequency' => $employee->activeContract->payment_frequency ?? 'N/A'
            ];
        });

        $years = range(date('Y') - 1, date('Y') + 1);
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];

        return view('payroll.create', compact('employees', 'employeesForJs', 'years', 'months'));
    }

    /**
     * Store individual payroll
     */
    public function store(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'year' => 'required|integer|min:2020|max:2030',
            'month' => 'required|integer|min:1|max:12',
        ]);

        try {
            $employee = Employee::findOrFail($request->employee_id);
            $payroll = $this->payrollService->generateEmployeePayroll(
                $employee,
                $request->year,
                $request->month
            );

            return redirect()->route('payroll.show', $payroll)
                ->with('success', 'Payroll generated successfully.');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to generate payroll: ' . $e->getMessage()]);
        }
    }

    /**
     * Show payroll details
     */
    public function show(Payroll $payroll)
    {
        $payroll->load(['employee', 'contract', 'attendances', 'approvals.approver']);
        
        // Get tax rules for this employment type
        $taxRules = KenyaTaxCalculator::getEmploymentTypeTaxRules($payroll->employment_type);
        
        // Get approval progress
        $approvalProgress = $this->approvalService->getApprovalProgress($payroll);
        $canApprove = $this->approvalService->canUserApprove($payroll, auth()->user());
        
        return view('payroll.show', compact('payroll', 'taxRules', 'approvalProgress', 'canApprove'));
    }

    /**
     * Approve payroll (hierarchical approval)
     */
    public function approve(Payroll $payroll)
    {
        $this->authorize('approve', $payroll);

        try {
            $this->approvalService->approveCurrentLevel($payroll, auth()->user());
            return back()->with('success', 'Payroll approved successfully.');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to approve payroll: ' . $e->getMessage()]);
        }
    }

    /**
     * Reject payroll
     */
    public function reject(Payroll $payroll, Request $request)
    {
        $this->authorize('reject', $payroll);

        $request->validate([
            'comments' => 'required|string|max:500',
        ]);

        try {
            $this->approvalService->rejectCurrentLevel($payroll, auth()->user(), $request->comments);
            return back()->with('success', 'Payroll rejected.');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to reject payroll: ' . $e->getMessage()]);
        }
    }

    /**
     * Process payroll (approved -> processed)
     */
    public function process(Payroll $payroll)
    {
        $this->authorize('process', $payroll);

        try {
            $this->payrollService->processPayroll($payroll, auth()->id());
            return back()->with('success', 'Payroll processed successfully and ready for payment.');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to process payroll: ' . $e->getMessage()]);
        }
    }

    /**
     * Mark payroll as paid
     */
    public function markPaid(Request $request, Payroll $payroll)
    {
        $this->authorize('markPaid', $payroll);

        $request->validate([
            'payment_date' => 'required|date',
            'payment_method' => 'required|string|in:bank_transfer,cash,cheque,mobile_money',
        ]);

        $payroll->update([
            'status' => 'paid',
            'payment_date' => $request->payment_date,
            'payment_method' => $request->payment_method,
        ]);

        return back()->with('success', 'Payroll marked as paid successfully.');
    }

    /**
     * Delete payroll
     */
    public function destroy(Payroll $payroll)
    {
        $this->authorize('delete', $payroll);

        $payroll->delete();

        return redirect()->route('payroll.index')
            ->with('success', 'Payroll deleted successfully.');
    }

    /**
     * Show payroll calculator/preview page
     */
    public function calculator()
    {
        $employmentTypes = [
            'permanent' => 'Permanent Employee',
            'contract' => 'Contract Worker',
            'freelance' => 'Freelancer',
            'casual' => 'Casual Worker',
            'intern' => 'Intern',
            'part_time' => 'Part-time Worker'
        ];

        return view('payroll.calculator', compact('employmentTypes'));
    }

    /**
     * Calculate payroll preview (AJAX)
     */
    public function calculatePreview(Request $request)
    {
        try {
            \Log::info('Calculate Preview Request', $request->all());
            
            $request->validate([
                'employment_type' => 'required|string|in:permanent,contract,freelance,casual,intern,part_time',
                'basic_salary' => 'required|numeric|min:0',
                'house_allowance' => 'nullable|numeric|min:0',
                'transport_allowance' => 'nullable|numeric|min:0',
                'medical_allowance' => 'nullable|numeric|min:0',
                'other_allowances' => 'nullable|numeric|min:0',
                'bonuses' => 'nullable|numeric|min:0',
                'overtime_pay' => 'nullable|numeric|min:0',
            ]);

            $salaryComponents = $request->only([
                'basic_salary', 'house_allowance', 'transport_allowance', 
                'medical_allowance', 'other_allowances', 'bonuses', 'overtime_pay'
            ]);

            // Convert empty strings to 0
            foreach ($salaryComponents as $key => $value) {
                $salaryComponents[$key] = $value ?: 0;
            }

            \Log::info('Salary Components', $salaryComponents);

            $calculations = KenyaTaxCalculator::calculatePayrollByEmploymentType(
                $salaryComponents,
                $request->employment_type
            );

            $taxRules = KenyaTaxCalculator::getEmploymentTypeTaxRules($request->employment_type);

            \Log::info('Calculations successful', ['calculations' => $calculations, 'tax_rules' => $taxRules]);

            return response()->json([
                'employment_type' => $request->employment_type,
                'calculations' => $calculations,
                'tax_rules' => $taxRules,
            ]);
        } catch (\Exception $e) {
            \Log::error('Calculate Preview Error', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'error' => 'An error occurred while calculating payroll. Please try again.',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}