<?php

namespace App\Http\Controllers;

use App\Models\TrainingSession;
use App\Models\TrainingProgram;
use App\Models\TrainingEnrollment;
use App\Models\User;
use App\Models\Employee;
use Illuminate\Http\Request;
use Carbon\Carbon;

class TrainingSessionController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', TrainingSession::class);

        $query = TrainingSession::with(['program.category', 'instructor']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->filled('from_date')) {
            $query->where('start_datetime', '>=', Carbon::parse($request->from_date));
        }

        if ($request->filled('to_date')) {
            $query->where('start_datetime', '<=', Carbon::parse($request->to_date)->endOfDay());
        }

        // Filter by program
        if ($request->filled('program')) {
            $query->where('training_program_id', $request->program);
        }

        $sessions = $query->orderBy('start_datetime', 'desc')->paginate(15);
        $programs = TrainingProgram::active()->get();

        return view('training.sessions.index', compact('sessions', 'programs'));
    }

    public function create(Request $request)
    {
        $this->authorize('create', TrainingSession::class);

        $programs = TrainingProgram::active()->get();
        $instructors = User::whereHas('employee')->get();
        
        $selectedProgram = null;
        if ($request->filled('program_id')) {
            $selectedProgram = TrainingProgram::find($request->program_id);
        }

        return view('training.sessions.create', compact('programs', 'instructors', 'selectedProgram'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', TrainingSession::class);

        $validated = $request->validate([
            'training_program_id' => 'required|exists:training_programs,id',
            'start_datetime' => 'required|date|after:now',
            'end_datetime' => 'required|date|after:start_datetime',
            'location' => 'nullable|string|max:255',
            'virtual_meeting_link' => 'nullable|url',
            'session_notes' => 'nullable|string',
            'max_participants' => 'nullable|integer|min:1',
            'instructor_id' => 'nullable|exists:users,id',
        ]);

        // Generate unique session code
        $validated['session_code'] = $this->generateSessionCode();
        $validated['created_by'] = auth()->id();
        $validated['organization_id'] = auth()->user()->organization_id;

        $session = TrainingSession::create($validated);

        return redirect()->route('training.sessions.show', $session)
            ->with('success', 'Training session created successfully!');
    }

    public function show(TrainingSession $session)
    {
        $this->authorize('view', $session);

        $session->load(['program.category', 'instructor', 'enrollments.employee']);

        // Get enrollment statistics
        $enrollmentStats = [
            'total' => $session->enrollments()->count(),
            'enrolled' => $session->enrollments()->where('status', 'enrolled')->count(),
            'attended' => $session->enrollments()->where('status', 'attended')->count(),
            'completed' => $session->enrollments()->where('status', 'completed')->count(),
            'no_show' => $session->enrollments()->where('status', 'no_show')->count(),
        ];

        return view('training.sessions.show', compact('session', 'enrollmentStats'));
    }

    public function edit(TrainingSession $session)
    {
        $this->authorize('update', $session);

        $programs = TrainingProgram::active()->get();
        $instructors = User::whereHas('employee')->get();

        return view('training.sessions.edit', compact('session', 'programs', 'instructors'));
    }

    public function update(Request $request, TrainingSession $session)
    {
        $this->authorize('update', $session);

        $validated = $request->validate([
            'training_program_id' => 'required|exists:training_programs,id',
            'start_datetime' => 'required|date',
            'end_datetime' => 'required|date|after:start_datetime',
            'location' => 'nullable|string|max:255',
            'virtual_meeting_link' => 'nullable|url',
            'session_notes' => 'nullable|string',
            'max_participants' => 'nullable|integer|min:1',
            'instructor_id' => 'nullable|exists:users,id',
            'status' => 'required|in:scheduled,in_progress,completed,cancelled',
        ]);

        $session->update($validated);

        return redirect()->route('training.sessions.show', $session)
            ->with('success', 'Training session updated successfully!');
    }

    public function destroy(TrainingSession $session)
    {
        $this->authorize('delete', $session);

        // Check if session has enrollments
        if ($session->enrollments()->count() > 0) {
            return back()->with('error', 'Cannot delete session with enrollments.');
        }

        $session->delete();

        return redirect()->route('training.sessions.index')
            ->with('success', 'Training session deleted successfully!');
    }

    public function enrollEmployee(Request $request, TrainingSession $session)
    {
        $this->authorize('update', $session);

        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'enrollment_type' => 'required|in:mandatory,voluntary,manager_assigned',
        ]);

        // Check if already enrolled
        $existingEnrollment = TrainingEnrollment::where('training_session_id', $session->id)
            ->where('employee_id', $validated['employee_id'])
            ->first();

        if ($existingEnrollment) {
            return back()->with('error', 'Employee is already enrolled in this session.');
        }

        // Check capacity
        if ($session->is_full) {
            return back()->with('error', 'This training session is full.');
        }

        TrainingEnrollment::create([
            'training_session_id' => $session->id,
            'employee_id' => $validated['employee_id'],
            'enrollment_type' => $validated['enrollment_type'],
            'status' => 'enrolled',
            'enrolled_at' => now(),
            'enrolled_by' => auth()->id(),
        ]);

        return back()->with('success', 'Employee enrolled successfully!');
    }

    public function updateAttendance(Request $request, TrainingSession $session)
    {
        $this->authorize('update', $session);

        $validated = $request->validate([
            'attendances' => 'required|array',
            'attendances.*' => 'in:enrolled,attended,completed,no_show',
        ]);

        foreach ($validated['attendances'] as $enrollmentId => $status) {
            $enrollment = TrainingEnrollment::find($enrollmentId);
            if ($enrollment && $enrollment->training_session_id == $session->id) {
                $enrollment->update([
                    'status' => $status,
                    'attended_at' => in_array($status, ['attended', 'completed']) ? now() : null,
                    'completed_at' => $status === 'completed' ? now() : null,
                ]);
            }
        }

        return back()->with('success', 'Attendance updated successfully!');
    }

    private function generateSessionCode()
    {
        do {
            $code = 'TS-' . strtoupper(substr(uniqid(), -6));
        } while (TrainingSession::where('session_code', $code)->exists());

        return $code;
    }
}