<?php

namespace App\Http\Controllers;

use App\Models\TrainingProgram;
use App\Models\TrainingCategory;
use App\Models\TrainingMaterial;
use App\Models\Department;
use App\Models\Position;
use App\Models\Skill;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;

class TrainingProgramController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', TrainingProgram::class);
        
        $query = TrainingProgram::with(['category', 'creator']);

        if ($request->filled('category')) {
            $query->where('training_category_id', $request->category);
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('status')) {
            $active = $request->status === 'active';
            $query->where('is_active', $active);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $programs = $query->orderBy('created_at', 'desc')->paginate(15);
        $categories = TrainingCategory::active()->get();

        return view('training.programs.index', compact('programs', 'categories'));
    }

    public function create()
    {
        $this->authorize('create', TrainingProgram::class);
        
        $categories = TrainingCategory::active()->get();
        $departments = Department::where('active', true)->get();
        $positions = Position::where('active', true)->get();
        $skills = Skill::active()->get();

        return view('training.programs.create', compact('categories', 'departments', 'positions', 'skills'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', TrainingProgram::class);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'training_category_id' => 'required|exists:training_categories,id',
            'type' => 'required|in:internal,external,online,workshop,seminar,certification',
            'delivery_method' => 'required|in:in_person,virtual,hybrid,self_paced',
            'duration_hours' => 'required|numeric|min:0.5|max:1000',
            'cost_per_participant' => 'nullable|numeric|min:0',
            'max_participants' => 'nullable|integer|min:1',
            'prerequisites' => 'nullable|array',
            'learning_objectives' => 'nullable|array',
            'content_outline' => 'nullable|string',
            'instructor_name' => 'nullable|string|max:255',
            'instructor_email' => 'nullable|email',
            'provider_name' => 'nullable|string|max:255',
            'difficulty_level' => 'required|in:beginner,intermediate,advanced',
            'target_roles' => 'nullable|array',
            'is_mandatory' => 'boolean',
        ]);

        $validated['created_by'] = Auth::id();
        $validated['organization_id'] = Auth::user()->organization_id;

        $program = TrainingProgram::create($validated);

        return redirect()->route('training.programs.show', $program)
            ->with('success', 'Training program created successfully!');
    }

    public function show(TrainingProgram $program)
    {
        $this->authorize('view', $program);

        $program->load(['category', 'sessions', 'materials', 'creator']);

        // Get enrollment statistics
        $enrollmentStats = [
            'total_enrollments' => $program->enrollments()->count(),
            'completed' => $program->enrollments()->where('training_enrollments.status', 'completed')->count(),
            'in_progress' => $program->enrollments()->whereIn('training_enrollments.status', ['enrolled', 'attended'])->count(),
            'completion_rate' => 0,
        ];

        if ($enrollmentStats['total_enrollments'] > 0) {
            $enrollmentStats['completion_rate'] = round(
                ($enrollmentStats['completed'] / $enrollmentStats['total_enrollments']) * 100, 1
            );
        }

        return view('training.programs.show', compact('program', 'enrollmentStats'));
    }

    public function edit(TrainingProgram $program)
    {
        $this->authorize('update', $program);

        $categories = TrainingCategory::active()->get();
        $departments = Department::where('active', true)->get();
        $positions = Position::where('active', true)->get();
        $skills = Skill::active()->get();

        return view('training.programs.edit', compact('program', 'categories', 'departments', 'positions', 'skills'));
    }

    public function update(Request $request, TrainingProgram $program)
    {
        $this->authorize('update', $program);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'training_category_id' => 'required|exists:training_categories,id',
            'type' => 'required|in:internal,external,online,workshop,seminar,certification',
            'delivery_method' => 'required|in:in_person,virtual,hybrid,self_paced',
            'duration_hours' => 'required|numeric|min:0.5|max:1000',
            'cost_per_participant' => 'nullable|numeric|min:0',
            'max_participants' => 'nullable|integer|min:1',
            'prerequisites' => 'nullable|array',
            'learning_objectives' => 'nullable|array',
            'content_outline' => 'nullable|string',
            'instructor_name' => 'nullable|string|max:255',
            'instructor_email' => 'nullable|email',
            'provider_name' => 'nullable|string|max:255',
            'difficulty_level' => 'required|in:beginner,intermediate,advanced',
            'target_roles' => 'nullable|array',
            'is_mandatory' => 'boolean',
            'is_active' => 'boolean',
        ]);

        $program->update($validated);

        return redirect()->route('training.programs.show', $program)
            ->with('success', 'Training program updated successfully!');
    }

    public function destroy(TrainingProgram $program)
    {
        $this->authorize('delete', $program);

        // Check if program has active sessions
        $activeSessions = $program->sessions()->whereIn('status', ['scheduled', 'in_progress'])->count();
        if ($activeSessions > 0) {
            return back()->with('error', 'Cannot delete program with active sessions.');
        }

        $program->delete();

        return redirect()->route('training.programs.index')
            ->with('success', 'Training program deleted successfully!');
    }

    public function addMaterial(Request $request, TrainingProgram $program)
    {
        $this->authorize('update', $program);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:document,video,presentation,link,quiz,assignment',
            'file' => 'nullable|file|max:10240', // 10MB max
            'external_url' => 'nullable|url',
            'is_required' => 'boolean',
            'is_downloadable' => 'boolean',
        ]);

        // Handle file upload
        if ($request->hasFile('file')) {
            $validated['file_path'] = $request->file('file')->store('training-materials', 'public');
        }

        // Set order index
        $validated['order_index'] = $program->materials()->max('order_index') + 1;
        $validated['training_program_id'] = $program->id;

        TrainingMaterial::create($validated);

        return back()->with('success', 'Training material added successfully!');
    }

    public function removeMaterial(TrainingProgram $program, TrainingMaterial $material)
    {
        $this->authorize('update', $program);

        // Delete file if exists
        if ($material->file_path && Storage::disk('public')->exists($material->file_path)) {
            Storage::disk('public')->delete($material->file_path);
        }

        $material->delete();

        return back()->with('success', 'Training material removed successfully!');
    }
}