<?php

namespace App\Http\Controllers;

use App\Models\TrainingPerformanceImpact;
use App\Models\TrainingProgram;
use App\Models\Employee;
use App\Models\PerformanceReview;
use App\Models\TrainingEnrollment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class TrainingPerformanceController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', TrainingPerformanceImpact::class);

        $query = TrainingPerformanceImpact::with(['trainingProgram', 'employee', 'performanceReview', 'measuredBy']);

        // Filter by training program
        if ($request->filled('training_program')) {
            $query->where('training_program_id', $request->training_program);
        }

        // Filter by employee
        if ($request->filled('employee')) {
            $query->where('employee_id', $request->employee);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->where('training_completed_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->where('training_completed_date', '<=', $request->date_to);
        }

        // Filter by impact type
        if ($request->filled('impact_type')) {
            switch ($request->impact_type) {
                case 'positive':
                    $query->positiveImpact();
                    break;
                case 'negative':
                    $query->whereRaw('performance_score_after < performance_score_before');
                    break;
                case 'neutral':
                    $query->whereRaw('performance_score_after = performance_score_before');
                    break;
            }
        }

        $impacts = $query->orderBy('training_completed_date', 'desc')->paginate(15);

        // Statistics
        $stats = $this->calculateImpactStatistics();

        $trainingPrograms = TrainingProgram::where('is_active', true)->get();
        $employees = Employee::where('status', 'active')->get();

        return view('training.performance.index', compact('impacts', 'stats', 'trainingPrograms', 'employees'));
    }

    public function create(Request $request)
    {
        $this->authorize('create', TrainingPerformanceImpact::class);

        $employees = Employee::where('status', 'active')->get();
        $trainingPrograms = TrainingProgram::where('is_active', true)->get();
        $performanceReviews = PerformanceReview::with('employee')->where('status', 'completed')->get();

        // Pre-select if provided in request
        $selectedEmployee = $request->filled('employee_id') ? Employee::find($request->employee_id) : null;
        $selectedProgram = $request->filled('training_program_id') ? TrainingProgram::find($request->training_program_id) : null;

        return view('training.performance.create', compact(
            'employees', 
            'trainingPrograms', 
            'performanceReviews',
            'selectedEmployee',
            'selectedProgram'
        ));
    }

    public function store(Request $request)
    {
        $this->authorize('create', TrainingPerformanceImpact::class);

        $validated = $request->validate([
            'training_program_id' => 'required|exists:training_programs,id',
            'employee_id' => 'required|exists:employees,id',
            'performance_review_id' => 'nullable|exists:performance_reviews,id',
            'performance_score_before' => 'required|numeric|min:0|max:5',
            'performance_score_after' => 'required|numeric|min:0|max:5',
            'training_completed_date' => 'required|date',
            'performance_measured_date' => 'required|date|after_or_equal:training_completed_date',
            'skill_improvements' => 'nullable|array',
            'impact_notes' => 'nullable|string|max:1000',
        ]);

        $validated['measured_by'] = Auth::id();

        TrainingPerformanceImpact::create($validated);

        return redirect()->route('training.performance.index')
            ->with('success', 'Training performance impact recorded successfully!');
    }

    public function show(TrainingPerformanceImpact $impact)
    {
        $this->authorize('view', $impact);

        $impact->load(['trainingProgram.category', 'employee', 'performanceReview', 'measuredBy']);

        return view('training.performance.show', compact('impact'));
    }

    public function edit(TrainingPerformanceImpact $impact)
    {
        $this->authorize('update', $impact);

        $employees = Employee::where('status', 'active')->get();
        $trainingPrograms = TrainingProgram::where('is_active', true)->get();
        $performanceReviews = PerformanceReview::with('employee')->where('status', 'completed')->get();

        return view('training.performance.edit', compact('impact', 'employees', 'trainingPrograms', 'performanceReviews'));
    }

    public function update(Request $request, TrainingPerformanceImpact $impact)
    {
        $this->authorize('update', $impact);

        $validated = $request->validate([
            'performance_score_before' => 'required|numeric|min:0|max:5',
            'performance_score_after' => 'required|numeric|min:0|max:5',
            'training_completed_date' => 'required|date',
            'performance_measured_date' => 'required|date|after_or_equal:training_completed_date',
            'skill_improvements' => 'nullable|array',
            'impact_notes' => 'nullable|string|max:1000',
        ]);

        $impact->update($validated);

        return redirect()->route('training.performance.show', $impact)
            ->with('success', 'Training performance impact updated successfully!');
    }

    public function destroy(TrainingPerformanceImpact $impact)
    {
        $this->authorize('delete', $impact);

        $impact->delete();

        return redirect()->route('training.performance.index')
            ->with('success', 'Training performance impact deleted successfully!');
    }

    public function analytics(Request $request)
    {
        $this->authorize('viewAny', TrainingPerformanceImpact::class);

        // Training Program Effectiveness
        $programEffectiveness = TrainingProgram::select([
                'training_programs.id',
                'training_programs.title',
                DB::raw('COUNT(training_performance_impacts.id) as impact_count'),
                DB::raw('AVG(training_performance_impacts.performance_score_after - training_performance_impacts.performance_score_before) as avg_improvement'),
                DB::raw('COUNT(CASE WHEN training_performance_impacts.performance_score_after > training_performance_impacts.performance_score_before THEN 1 END) as positive_impacts'),
            ])
            ->leftJoin('training_performance_impacts', 'training_programs.id', '=', 'training_performance_impacts.training_program_id')
            ->leftJoin('employees', 'training_performance_impacts.employee_id', '=', 'employees.id')
            ->where('training_programs.is_active', true)
            ->where('training_programs.organization_id', Auth::user()->organization_id)
            ->where(function ($query) {
                $query->whereNull('employees.id')
                      ->orWhere('employees.organization_id', Auth::user()->organization_id);
            })
            ->groupBy('training_programs.id', 'training_programs.title')
            ->having('impact_count', '>', 0)
            ->orderBy('avg_improvement', 'desc')
            ->get();

        // Monthly Impact Trends
        $monthlyTrends = TrainingPerformanceImpact::select([
                DB::raw('DATE_FORMAT(training_completed_date, "%Y-%m") as month'),
                DB::raw('COUNT(*) as total_impacts'),
                DB::raw('AVG(performance_score_after - performance_score_before) as avg_improvement'),
                DB::raw('COUNT(CASE WHEN performance_score_after > performance_score_before THEN 1 END) as positive_impacts'),
            ])
            ->whereHas('employee', function ($query) {
                $query->where('organization_id', Auth::user()->organization_id);
            })
            ->where('training_completed_date', '>=', now()->subMonths(12))
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        // Top Performing Employees (Most Improved)
        $topImprovedEmployees = Employee::select([
                'employees.id',
                'employees.first_name',
                'employees.last_name',
                DB::raw('COUNT(training_performance_impacts.id) as training_count'),
                DB::raw('AVG(training_performance_impacts.performance_score_after - training_performance_impacts.performance_score_before) as avg_improvement'),
                DB::raw('SUM(training_performance_impacts.performance_score_after - training_performance_impacts.performance_score_before) as total_improvement'),
            ])
            ->join('training_performance_impacts', 'employees.id', '=', 'training_performance_impacts.employee_id')
            ->where('employees.organization_id', Auth::user()->organization_id)
            ->groupBy('employees.id', 'employees.first_name', 'employees.last_name')
            ->having('training_count', '>=', 2)
            ->orderBy('avg_improvement', 'desc')
            ->limit(10)
            ->get();

        // Skills Development Impact
        $skillsImpact = TrainingPerformanceImpact::whereNotNull('skill_improvements')
            ->whereHas('employee', function ($query) {
                $query->where('organization_id', Auth::user()->organization_id);
            })
            ->get()
            ->flatMap(function ($impact) {
                return collect($impact->skill_improvements)->map(function ($improvement, $skillName) use ($impact) {
                    return [
                        'skill_name' => $skillName,
                        'improvement' => $improvement,
                        'employee_id' => $impact->employee_id,
                        'training_program_id' => $impact->training_program_id,
                    ];
                });
            })
            ->groupBy('skill_name')
            ->map(function ($skillGroup, $skillName) {
                return [
                    'skill_name' => $skillName,
                    'total_improvements' => $skillGroup->count(),
                    'avg_improvement' => $skillGroup->avg('improvement'),
                    'employees_impacted' => $skillGroup->pluck('employee_id')->unique()->count(),
                ];
            })
            ->sortByDesc('avg_improvement')
            ->take(10);

        return view('training.performance.analytics', compact(
            'programEffectiveness',
            'monthlyTrends',
            'topImprovedEmployees',
            'skillsImpact'
        ));
    }

    public function autoTrack(Request $request)
    {
        $this->authorize('create', TrainingPerformanceImpact::class);

        $validated = $request->validate([
            'days_after_training' => 'required|integer|min:30|max:365',
        ]);

        $daysAfter = $validated['days_after_training'];
        $cutoffDate = now()->subDays($daysAfter);

        // Find completed training enrollments that don't have performance impact records
        $completedTrainings = TrainingEnrollment::where('status', 'completed')
            ->where('completed_at', '<=', $cutoffDate)
            ->whereDoesntHave('performanceImpacts')
            ->with(['employee', 'session.program'])
            ->get();

        $trackedCount = 0;

        foreach ($completedTrainings as $enrollment) {
            $employee = $enrollment->employee;
            
            // Get performance review before training
            $beforeReview = $employee->performanceReviews()
                ->where('status', 'completed')
                ->where('review_date', '<=', $enrollment->completed_at)
                ->orderBy('review_date', 'desc')
                ->first();

            // Get performance review after training
            $afterReview = $employee->performanceReviews()
                ->where('status', 'completed')
                ->where('review_date', '>=', $enrollment->completed_at->addDays($daysAfter))
                ->orderBy('review_date', 'asc')
                ->first();

            if ($beforeReview && $afterReview) {
                TrainingPerformanceImpact::create([
                    'training_program_id' => $enrollment->session->program->id,
                    'employee_id' => $employee->id,
                    'performance_review_id' => $afterReview->id,
                    'performance_score_before' => $beforeReview->overall_rating,
                    'performance_score_after' => $afterReview->overall_rating,
                    'training_completed_date' => $enrollment->completed_at->toDateString(),
                    'performance_measured_date' => $afterReview->review_date,
                    'impact_notes' => 'Auto-tracked performance impact',
                    'measured_by' => Auth::id(),
                ]);

                $trackedCount++;
            }
        }

        return back()->with('success', "Auto-tracked {$trackedCount} training performance impacts!");
    }

    private function calculateImpactStatistics()
    {
        $organizationId = Auth::user()->organization_id;
        
        return [
            'total_impacts' => TrainingPerformanceImpact::whereHas('employee', function ($query) use ($organizationId) {
                $query->where('organization_id', $organizationId);
            })->count(),
            'positive_impacts' => TrainingPerformanceImpact::positiveImpact()
                ->whereHas('employee', function ($query) use ($organizationId) {
                    $query->where('organization_id', $organizationId);
                })->count(),
            'avg_improvement' => TrainingPerformanceImpact::whereHas('employee', function ($query) use ($organizationId) {
                    $query->where('organization_id', $organizationId);
                })
                ->selectRaw('AVG(performance_score_after - performance_score_before) as avg')
                ->value('avg') ?? 0,
            'best_program' => TrainingProgram::select(['training_programs.title'])
                ->join('training_performance_impacts', 'training_programs.id', '=', 'training_performance_impacts.training_program_id')
                ->join('employees', 'training_performance_impacts.employee_id', '=', 'employees.id')
                ->where('employees.organization_id', $organizationId)
                ->selectRaw('AVG(performance_score_after - performance_score_before) as avg_improvement')
                ->groupBy('training_programs.id', 'training_programs.title')
                ->orderBy('avg_improvement', 'desc')
                ->first()?->title ?? 'N/A',
        ];
    }
}