<?php

namespace App\Http\Controllers;

use App\Models\TrainingProgram;
use App\Models\TrainingCategory;
use App\Models\TrainingSession;
use App\Models\TrainingEnrollment;
use App\Models\Employee;
use App\Models\Department;
use App\Models\Position;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class TrainingController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', TrainingProgram::class);

        $query = TrainingProgram::with(['category', 'creator', 'sessions'])
            ->where('is_active', true);

        // Filter by category
        if ($request->filled('category')) {
            $query->where('training_category_id', $request->category);
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by difficulty
        if ($request->filled('difficulty')) {
            $query->where('difficulty_level', $request->difficulty);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $programs = $query->orderBy('title')->paginate(12);
        $categories = TrainingCategory::active()->get();

        // Get user's enrollments for quick access
        $userEnrollments = [];
        if (auth()->user()->employee) {
            $userEnrollments = TrainingEnrollment::where('employee_id', auth()->user()->employee->id)
                ->whereHas('session.program')
                ->with('session.program')
                ->get()
                ->pluck('session.program.id')
                ->toArray();
        }

        return view('training.index', compact('programs', 'categories', 'userEnrollments'));
    }

    public function show(TrainingProgram $program)
    {
        $this->authorize('view', $program);

        $program->load(['category', 'sessions' => function($query) {
            $query->where('status', 'scheduled')
                  ->where('start_datetime', '>', now())
                  ->orderBy('start_datetime');
        }, 'materials' => function($query) {
            $query->ordered();
        }]);

        // Check if user is enrolled
        $userEnrollment = null;
        if (auth()->user()->employee) {
            $userEnrollment = TrainingEnrollment::where('employee_id', auth()->user()->employee->id)
                ->whereHas('session', function($query) use ($program) {
                    $query->where('training_program_id', $program->id);
                })
                ->first();
        }

        return view('training.show', compact('program', 'userEnrollment'));
    }

    public function enroll(Request $request, TrainingSession $session)
    {
        $this->authorize('enroll', TrainingProgram::class);

        $employee = auth()->user()->employee;
        if (!$employee) {
            return back()->with('error', 'Employee record not found.');
        }

        // Check if session is full
        if ($session->is_full) {
            return back()->with('error', 'This training session is full.');
        }

        // Check if already enrolled
        $existingEnrollment = TrainingEnrollment::where('training_session_id', $session->id)
            ->where('employee_id', $employee->id)
            ->first();

        if ($existingEnrollment) {
            return back()->with('error', 'You are already enrolled in this session.');
        }

        // Create enrollment
        TrainingEnrollment::create([
            'training_session_id' => $session->id,
            'employee_id' => $employee->id,
            'enrollment_type' => 'voluntary',
            'status' => 'enrolled',
            'enrolled_at' => now(),
            'enrolled_by' => auth()->id(),
        ]);

        return back()->with('success', 'Successfully enrolled in the training session!');
    }

    public function myTrainings()
    {
        $employee = auth()->user()->employee;
        if (!$employee) {
            return redirect()->route('training.index')->with('error', 'Employee record not found.');
        }

        $enrollments = TrainingEnrollment::where('employee_id', $employee->id)
            ->with(['session.program.category', 'session.instructor'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        // Get upcoming sessions
        $upcomingSessions = TrainingEnrollment::where('employee_id', $employee->id)
            ->whereIn('status', ['enrolled'])
            ->whereHas('session', function($query) {
                $query->where('start_datetime', '>', now())
                      ->where('status', 'scheduled');
            })
            ->with(['session.program', 'session.instructor'])
            ->get();

        // Get completed trainings
        $completedTrainings = TrainingEnrollment::where('employee_id', $employee->id)
            ->where('status', 'completed')
            ->with(['session.program.category'])
            ->orderBy('completed_at', 'desc')
            ->limit(5)
            ->get();

        return view('training.my-trainings', compact('enrollments', 'upcomingSessions', 'completedTrainings'));
    }

    public function dashboard()
    {
        // Check if user can view training reports/dashboard
        if (!auth()->user()->can('view_training_reports') && !auth()->user()->can('manage_training')) {
            abort(403, 'Unauthorized to view training dashboard.');
        }

        // Training statistics
        $stats = [
            'total_programs' => TrainingProgram::where('is_active', true)->count(),
            'active_sessions' => TrainingSession::whereIn('status', ['scheduled', 'in_progress'])->count(),
            'total_enrollments' => TrainingEnrollment::count(),
            'completion_rate' => $this->calculateCompletionRate(),
        ];

        // Recent activities
        $recentEnrollments = TrainingEnrollment::with(['employee', 'session.program'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Upcoming sessions
        $upcomingSessions = TrainingSession::with(['program', 'instructor'])
            ->where('status', 'scheduled')
            ->where('start_datetime', '>', now())
            ->orderBy('start_datetime')
            ->limit(5)
            ->get();

        // Popular programs
        $popularPrograms = TrainingProgram::withCount('enrollments')
            ->where('is_active', true)
            ->orderBy('enrollments_count', 'desc')
            ->limit(5)
            ->get();

        return view('training.dashboard', compact('stats', 'recentEnrollments', 'upcomingSessions', 'popularPrograms'));
    }

    private function calculateCompletionRate()
    {
        $totalEnrollments = TrainingEnrollment::count();
        if ($totalEnrollments === 0) return 0;

        $completedEnrollments = TrainingEnrollment::where('status', 'completed')->count();
        return round(($completedEnrollments / $totalEnrollments) * 100, 1);
    }
}