<?php

namespace App\Http\Controllers;

use App\Models\Organization;
use App\Models\User;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class SytemAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('system_admin')->except(['showLogin', 'login']);
    }

    /**
     * Show SytemAdmin login form
     */
    public function showLogin()
    {
        if (Auth::check() && Auth::user()->isSystemAdmin()) {
            return redirect()->route('systemadmin.dashboard');
        }

        return view('systemadmin.auth.login');
    }

    /**
     * Handle SytemAdmin login with enhanced security
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|min:8',
        ]);

        $credentials = $request->only('email', 'password');

        // SECURITY: Rate limiting - Stricter for system admin (3 attempts per 15 minutes per IP)
        $key = 'sysadmin-login-attempts-' . $request->ip();
        $attempts = cache()->get($key, 0);
        
        if ($attempts >= 3) {
            $lockoutTime = cache()->get($key . '-lockout', now());
            if (now()->lessThan($lockoutTime)) {
                Log::warning('System Admin login rate limit exceeded', [
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                    'attempts' => $attempts
                ]);

                return back()->withErrors([
                    'email' => 'Too many login attempts. Please try again in 15 minutes.',
                ])->withInput($request->only('email'));
            }
        }

        // Check if user exists
        $user = User::where('email', $credentials['email'])->first();

        // SECURITY: Check if account is locked
        if ($user && $user->locked_until && now()->lessThan($user->locked_until)) {
            Log::warning('System Admin login attempt on locked account', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'locked_until' => $user->locked_until
            ]);

            return back()->withErrors([
                'email' => 'Account is temporarily locked due to suspicious activity. Please contact support.',
            ])->withInput($request->only('email'));
        }

        // SECURITY: Verify system admin privileges before authentication
        if ($user && !$user->is_system_admin) {
            Log::warning('Non-system admin attempted system admin login', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]);

            // Increment attempts for non-admin trying to access
            $attempts++;
            cache()->put($key, $attempts, now()->addMinutes(15));

            return back()->withErrors([
                'email' => 'Access denied. System admin privileges required.',
            ])->withInput($request->only('email'));
        }

        // Attempt authentication
        if (Auth::attempt($credentials, $request->filled('remember'))) {
            $user = Auth::user();
            
            // Double-check system admin status
            if (!$user->is_system_admin) {
                Auth::logout();
                
                Log::critical('System admin check bypassed - security breach attempt', [
                    'user_id' => $user->id,
                    'email' => $user->email,
                    'ip_address' => $request->ip()
                ]);

                throw ValidationException::withMessages([
                    'email' => ['Access denied. System admin privileges required.'],
                ]);
            }

            // SECURITY: Clear failed attempts on successful login
            cache()->forget($key);
            cache()->forget($key . '-lockout');
            
            // SECURITY: Check if 2FA is enabled (mandatory for system admins)
            if ($user->two_factor_secret && $user->two_factor_confirmed_at) {
                // Store user ID and remember preference in session for 2FA
                session([
                    '2fa_user_id' => $user->id,
                    '2fa_remember' => $request->filled('remember'),
                    '2fa_intended' => route('systemadmin.dashboard')
                ]);
                
                // Logout temporarily until 2FA is verified
                Auth::logout();
                
                Log::info('System Admin 2FA challenge initiated', [
                    'user_id' => $user->id,
                    'email' => $user->email,
                    'ip_address' => $request->ip()
                ]);
                
                return redirect()->route('two-factor.challenge');
            }

            // SECURITY: Regenerate session to prevent session fixation
            $request->session()->regenerate();
            
            // Update last login timestamp
            $user->update([
                'last_login_at' => now(),
                'last_login_ip' => $request->ip()
            ]);
            
            // SECURITY: Comprehensive audit logging
            Log::info('System Admin logged in successfully', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'remember' => $request->filled('remember')
            ]);

            // TODO: Send email notification about system admin login
            // dispatch(new SystemAdminLoginNotification($user, $request->ip()));

            return redirect()->intended(route('systemadmin.dashboard'))
                ->with('success', 'Welcome back, System Administrator!');
        }

        // SECURITY: Increment failed attempts
        $attempts++;
        cache()->put($key, $attempts, now()->addMinutes(15));
        
        // SECURITY: Lock account after 5 failed attempts (stricter than org login)
        if ($user && $attempts >= 5) {
            $user->update(['locked_until' => now()->addHours(2)]); // 2 hour lockout
            cache()->put($key . '-lockout', now()->addMinutes(15), now()->addMinutes(15));
            
            Log::critical('System Admin account locked due to failed login attempts', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'attempts' => $attempts,
                'locked_until' => now()->addHours(2)
            ]);

            // TODO: Send email notification about account lock
            // dispatch(new SystemAdminAccountLockedNotification($user, $request->ip()));
        }

        // SECURITY: Log failed login attempt
        if ($user) {
            Log::warning('System Admin failed login attempt', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'attempts' => $attempts
            ]);
        } else {
            Log::warning('System Admin login attempt with non-existent email', [
                'email' => $credentials['email'],
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]);
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ])->withInput($request->only('email'));
    }

    /**
     * SytemAdmin dashboard
     */
    public function dashboard()
    {
        $stats = [
            'total_organizations' => Organization::count(),
            'active_organizations' => Organization::where('status', 'active')->count(),
            'total_users' => User::whereNotNull('organization_id')->count(),
            'total_employees' => Employee::withoutOrganizationScope()->count(),
            'recent_organizations' => Organization::latest()->take(5)->get(),
        ];

        return view('systemadmin.dashboard', compact('stats'));
    }

    /**
     * Handle SytemAdmin logout
     */
    public function logout(Request $request)
    {
        $user = Auth::user();
        
        // SECURITY: Log logout
        if ($user) {
            Log::info('System Admin logged out', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip()
            ]);
        }

        Auth::logout();
        
        // SECURITY: Invalidate session and regenerate token
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        return redirect()->route('systemadmin.login')
            ->with('success', 'You have been logged out successfully.');
    }
}
