<?php

namespace App\Http\Controllers;

use App\Models\LeaveType;
use App\Models\Position;
use App\Models\Holiday;
use App\Models\AttendanceSetting;
use Illuminate\Http\Request;

class SettingsController extends Controller
{
    public function index()
    {
        // Check if user has any settings permissions
        if (!auth()->user()->can('view_settings') && 
            !auth()->user()->can('edit_settings') && 
            !auth()->user()->can('view_organization_settings') && 
            !auth()->user()->can('manage_roles') && 
            !auth()->user()->can('view_users') &&
            !auth()->user()->can('edit_departments')) {
            abort(403, 'You do not have permission to access settings.');
        }
        
        return view('settings.index');
    }

    // Leave Types Management
    public function leaveTypes()
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to manage leave types.');
        }
        
        $leaveTypes = LeaveType::orderBy('name')->get();
        return view('settings.leave-types', compact('leaveTypes'));
    }

    public function storeLeaveType(Request $request)
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to create leave types.');
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'days_allowed' => 'required|integer|min:0',
            'requires_approval' => 'nullable|boolean',
            'is_paid' => 'required|boolean',
            'color' => 'nullable|string|max:7',
            'description' => 'nullable|string',
            'active' => 'nullable|boolean',
        ]);

        LeaveType::create($validated);

        return back()->with('success', 'Leave type created successfully.');
    }

    public function updateLeaveType(Request $request, LeaveType $leaveType)
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to update leave types.');
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'days_allowed' => 'required|integer|min:0',
            'requires_approval' => 'nullable|boolean',
            'is_paid' => 'required|boolean',
            'color' => 'nullable|string|max:7',
            'description' => 'nullable|string',
            'active' => 'nullable|boolean',
        ]);

        $leaveType->update($validated);

        return back()->with('success', 'Leave type updated successfully.');
    }

    public function deleteLeaveType(LeaveType $leaveType)
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to delete leave types.');
        }
        
        $leaveType->delete();
        return back()->with('success', 'Leave type deleted successfully.');
    }

    // Positions Management
    public function positions()
    {
        if (!auth()->user()->can('edit_departments')) {
            abort(403, 'You do not have permission to manage positions.');
        }
        
        $positions = Position::orderBy('name')->get();
        return view('settings.positions', compact('positions'));
    }

    public function storePosition(Request $request)
    {
        if (!auth()->user()->can('edit_departments')) {
            abort(403, 'You do not have permission to create positions.');
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'requirements' => 'nullable|string',
            'expected_employees' => 'nullable|integer|min:1',
            'active' => 'nullable|boolean',
        ]);

        Position::create($validated);

        return back()->with('success', 'Position created successfully.');
    }

    public function updatePosition(Request $request, Position $position)
    {
        if (!auth()->user()->can('edit_departments')) {
            abort(403, 'You do not have permission to update positions.');
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'requirements' => 'nullable|string',
            'expected_employees' => 'nullable|integer|min:1',
            'active' => 'nullable|boolean',
        ]);

        $position->update($validated);

        return back()->with('success', 'Position updated successfully.');
    }

    public function deletePosition(Position $position)
    {
        if (!auth()->user()->can('edit_departments')) {
            abort(403, 'You do not have permission to delete positions.');
        }
        
        $position->delete();
        return back()->with('success', 'Position deleted successfully.');
    }

    // Holidays Management
    public function holidays()
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to manage holidays.');
        }
        
        $holidays = Holiday::orderBy('date')->get();
        return view('settings.holidays', compact('holidays'));
    }

    public function storeHoliday(Request $request)
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to create holidays.');
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'date' => 'required|date',
            'type' => 'required|in:public,company',
            'description' => 'nullable|string',
        ]);

        Holiday::create($validated);

        return back()->with('success', 'Holiday created successfully.');
    }

    public function updateHoliday(Request $request, Holiday $holiday)
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to update holidays.');
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'date' => 'required|date',
            'type' => 'required|in:public,company',
            'description' => 'nullable|string',
        ]);

        $holiday->update($validated);

        return back()->with('success', 'Holiday updated successfully.');
    }

    public function deleteHoliday(Holiday $holiday)
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to delete holidays.');
        }
        
        $holiday->delete();
        return back()->with('success', 'Holiday deleted successfully.');
    }

    // Attendance Settings
    public function attendance()
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to manage attendance settings.');
        }
        
        $settings = AttendanceSetting::getSettings();
        return view('settings.attendance', compact('settings'));
    }

    public function updateAttendance(Request $request)
    {
        if (!auth()->user()->can('edit_settings')) {
            abort(403, 'You do not have permission to update attendance settings.');
        }
        
        $validated = $request->validate([
            'check_in_start' => 'required|date_format:H:i',
            'check_in_end' => 'required|date_format:H:i|after:check_in_start',
            'check_out_start' => 'required|date_format:H:i',
            'check_out_end' => 'required|date_format:H:i|after:check_out_start',
            'work_start_time' => 'required|date_format:H:i',
            'work_end_time' => 'required|date_format:H:i|after:work_start_time',
            'late_threshold_minutes' => 'required|integer|min:0|max:120',
            'early_leave_threshold_minutes' => 'required|integer|min:0|max:120',
            'half_day_hours' => 'required|integer|min:1|max:12',
            'full_day_hours' => 'required|integer|min:1|max:24',
            'allow_weekend_checkin' => 'boolean',
            'require_checkout' => 'boolean',
        ]);

        $settings = AttendanceSetting::first();
        $settings->update($validated);

        return back()->with('success', 'Attendance settings updated successfully.');
    }
}
