<?php

namespace App\Http\Controllers;

use App\Models\JobPosting;
use App\Models\Application;
use App\Mail\ApplicationReceived;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class PublicApplicationController extends Controller
{
    /**
     * Show the public application form
     */
    public function show($token)
    {
        $jobPosting = JobPosting::where('public_token', $token)
                               ->with(['department:id,name', 'position:id,name', 'organization:id,name'])
                               ->first();

        if (!$jobPosting) {
            abort(404, 'Job posting not found');
        }

        if (!$jobPosting->isActive()) {
            return view('recruitment.public.closed', compact('jobPosting'));
        }

        return view('recruitment.public.apply', compact('jobPosting'));
    }

    /**
     * Store a new application
     */
    public function store(Request $request, $token)
    {
        $jobPosting = JobPosting::where('public_token', $token)->first();

        if (!$jobPosting || !$jobPosting->isActive()) {
            return redirect()->back()
                           ->with('error', 'This job posting is no longer accepting applications.');
        }

        // Rate limiting check
        $this->checkRateLimit($request, $jobPosting);

        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'cv' => 'required|file|mimes:pdf,doc,docx|max:5120', // 5MB max
            'cover_letter' => 'nullable|string|max:2000',
            'g-recaptcha-response' => 'nullable', // Add reCAPTCHA if needed
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
        }

        // Check for duplicate application
        $existingApplication = Application::where('email', $request->email)
                                        ->where('job_posting_id', $jobPosting->id)
                                        ->first();

        if ($existingApplication) {
            // If application exists and is still in "new" status, check if it can be updated
            if ($existingApplication->status === 'new') {
                if ($existingApplication->canBeUpdated()) {
                    return view('recruitment.public.update-application', compact('existingApplication', 'jobPosting'));
                } else {
                    // Application has reached update limit
                    return view('recruitment.public.update-limit-reached', compact('existingApplication', 'jobPosting'));
                }
            } else {
                // Application already processed, show friendly message
                return view('recruitment.public.already-applied', compact('existingApplication', 'jobPosting'));
            }
        }

        // Store CV file
        $cvPath = $this->storeCvFile($request->file('cv'), $jobPosting);

        // Create application
        $application = Application::create([
            'job_posting_id' => $jobPosting->id,
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'email' => $request->email,
            'phone' => $request->phone,
            'cv_path' => $cvPath,
            'cover_letter' => $request->cover_letter,
            'status' => 'new',
            'organization_id' => $jobPosting->organization_id,
        ]);

        // Send confirmation email
        $this->sendConfirmationEmail($application, $jobPosting);

        return view('recruitment.public.success', compact('application', 'jobPosting'));
    }

    /**
     * Update an existing application (if status is still "new")
     */
    public function update(Request $request, $token)
    {
        $jobPosting = JobPosting::where('public_token', $token)->first();

        if (!$jobPosting || !$jobPosting->isActive()) {
            return redirect()->back()
                           ->with('error', 'This job posting is no longer accepting applications.');
        }

        // Find existing application
        $application = Application::where('email', $request->email)
                                ->where('job_posting_id', $jobPosting->id)
                                ->where('status', 'new') // Only allow updates for new applications
                                ->first();

        if (!$application) {
            return redirect()->back()
                           ->with('error', 'Application not found or cannot be updated.');
        }

        // Check if application can still be updated
        if (!$application->canBeUpdated()) {
            return redirect()->back()
                           ->with('error', 'This application has already been updated and cannot be modified further.');
        }

        // Rate limiting check
        $this->checkRateLimit($request, $jobPosting);

        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'phone' => 'required|string|max:20',
            'cv' => 'nullable|file|mimes:pdf,doc,docx|max:5120', // CV is optional for updates
            'cover_letter' => 'nullable|string|max:2000',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
        }

        // Update application data
        $updateData = [
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'phone' => $request->phone,
            'cover_letter' => $request->cover_letter,
            'updated_count' => $application->updated_count + 1,
            'last_updated_at' => now(),
        ];

        // Handle CV update if provided
        if ($request->hasFile('cv')) {
            // Delete old CV file
            if ($application->cv_path) {
                $oldPath = storage_path('app/public/' . str_replace('public/', '', $application->cv_path));
                if (file_exists($oldPath)) {
                    unlink($oldPath);
                }
            }
            
            // Store new CV
            $cvPath = $this->storeCvFile($request->file('cv'), $jobPosting);
            $updateData['cv_path'] = $cvPath;
        }

        $application->update($updateData);

        return view('recruitment.public.success', compact('application', 'jobPosting'))
               ->with('updated', true);
    }

    /**
     * Check rate limiting for applications
     */
    private function checkRateLimit(Request $request, JobPosting $jobPosting)
    {
        $key = 'application_rate_limit:' . $request->ip() . ':' . $request->email;
        $attempts = cache()->get($key, 0);

        if ($attempts >= 5) {
            abort(429, 'Too many application attempts. Please try again later.');
        }

        cache()->put($key, $attempts + 1, now()->addHour());
    }

    /**
     * Store CV file with organized structure
     */
    private function storeCvFile($file, JobPosting $jobPosting)
    {
        $filename = time() . '_' . $file->getClientOriginalName();
        $path = "applications/{$jobPosting->organization_id}/{$jobPosting->id}/{$filename}";
        
        // Store in public disk and return the path without 'public/' prefix
        $storedPath = $file->storeAs($path, $filename, 'public');
        
        return $storedPath;
    }

    /**
     * Send confirmation email to applicant
     */
    private function sendConfirmationEmail(Application $application, JobPosting $jobPosting)
    {
        try {
            Mail::to($application->email)->send(new ApplicationReceived($application, $jobPosting));
            \Log::info('Application confirmation email sent', [
                'application_id' => $application->id,
                'email' => $application->email
            ]);
        } catch (\Exception $e) {
            // Log error but don't fail the application
            \Log::error('Failed to send application confirmation email: ' . $e->getMessage(), [
                'application_id' => $application->id,
                'email' => $application->email,
                'error' => $e->getMessage()
            ]);
        }
    }
}