<?php

namespace App\Http\Controllers;

use App\Models\PerformanceGoal;
use App\Models\Employee;
use App\Models\PerformanceReview;
use Illuminate\Http\Request;

class PerformanceGoalController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', PerformanceGoal::class);
        
        $query = PerformanceGoal::with(['employee', 'createdBy']);

        // Search filters
        if ($request->filled('employee')) {
            $query->where('employee_id', $request->employee);
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('priority')) {
            $query->where('priority', $request->priority);
        }

        $goals = $query->orderBy('target_date', 'asc')->paginate(15);

        // Statistics
        $stats = [
            'total_goals' => PerformanceGoal::count(),
            'active_goals' => PerformanceGoal::where('status', 'active')->count(),
            'completed_goals' => PerformanceGoal::where('status', 'completed')->count(),
            'overdue_goals' => PerformanceGoal::overdue()->count(),
            'average_progress' => round(PerformanceGoal::where('status', 'active')->avg('progress_percentage'), 1),
        ];

        $employees = Employee::where('status', 'active')->get();

        return view('performance.goals.index', compact('goals', 'stats', 'employees'));
    }

    public function create(Request $request)
    {
        $this->authorize('create', PerformanceGoal::class);
        
        $employees = Employee::where('status', 'active')->get();
        $performanceReviews = PerformanceReview::with('employee')->get();
        
        $selectedEmployee = $request->employee_id ? Employee::find($request->employee_id) : null;

        return view('performance.goals.create', compact('employees', 'performanceReviews', 'selectedEmployee'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', PerformanceGoal::class);
        
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'performance_review_id' => 'nullable|exists:performance_reviews,id',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'type' => 'required|in:individual,team,company,development,project',
            'priority' => 'required|in:low,medium,high,critical',
            'start_date' => 'required|date',
            'target_date' => 'required|date|after:start_date',
            'weight_percentage' => 'required|integer|min:0|max:100',
            'target_value' => 'nullable|numeric|min:0',
            'measurement_unit' => 'nullable|string|max:50',
            'success_criteria' => 'required|string',
            'milestones' => 'nullable|array',
            'milestones.*.title' => 'required_with:milestones|string',
            'milestones.*.date' => 'required_with:milestones|date',
            'milestones.*.description' => 'nullable|string',
        ]);

        $validated['created_by'] = auth()->id();
        $validated['organization_id'] = auth()->user()->organization_id;
        $validated['status'] = 'active';

        $goal = PerformanceGoal::create($validated);

        return redirect()->route('performance.goals.show', $goal)
            ->with('success', 'Performance goal created successfully.');
    }

    public function show(PerformanceGoal $performanceGoal)
    {
        $this->authorize('view', $performanceGoal);
        
        $goal = $performanceGoal;
        $goal->load(['employee', 'createdBy', 'performanceReview', 'progressUpdates.updatedBy']);

        return view('performance.goals.show', compact('goal'));
    }

    public function edit(PerformanceGoal $performanceGoal)
    {
        $this->authorize('update', $performanceGoal);
        
        $goal = $performanceGoal;
        $employees = Employee::where('status', 'active')->get();
        $performanceReviews = PerformanceReview::with('employee')->get();

        return view('performance.goals.edit', compact('goal', 'employees', 'performanceReviews'));
    }

    public function update(Request $request, PerformanceGoal $performanceGoal)
    {
        $this->authorize('update', $performanceGoal);
        
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'performance_review_id' => 'nullable|exists:performance_reviews,id',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'type' => 'required|in:individual,team,company,development,project',
            'priority' => 'required|in:low,medium,high,critical',
            'start_date' => 'required|date',
            'target_date' => 'required|date|after:start_date',
            'weight_percentage' => 'required|integer|min:0|max:100',
            'target_value' => 'nullable|numeric|min:0',
            'measurement_unit' => 'nullable|string|max:50',
            'success_criteria' => 'required|string',
            'status' => 'required|in:draft,active,completed,cancelled',
            'milestones' => 'nullable|array',
            'milestones.*.title' => 'required_with:milestones|string',
            'milestones.*.date' => 'required_with:milestones|date',
            'milestones.*.description' => 'nullable|string',
        ]);

        $performanceGoal->update($validated);

        return redirect()->route('performance.goals.show', $performanceGoal)
            ->with('success', 'Performance goal updated successfully.');
    }

    public function updateProgress(Request $request, PerformanceGoal $performanceGoal)
    {
        $this->authorize('update', $performanceGoal);
        
        $validated = $request->validate([
            'progress_percentage' => 'required|numeric|min:0|max:100',
            'achieved_value' => 'nullable|numeric|min:0',
            'update_notes' => 'required|string',
        ]);

        $performanceGoal->updateProgress(
            $validated['progress_percentage'],
            $validated['achieved_value'] ?? null,
            $validated['update_notes'],
            auth()->id()
        );

        return redirect()->route('performance.goals.show', $performanceGoal)
            ->with('success', 'Goal progress updated successfully.');
    }

    public function destroy(PerformanceGoal $performanceGoal)
    {
        $this->authorize('delete', $performanceGoal);
        
        $performanceGoal->delete();

        return redirect()->route('performance.goals.index')
            ->with('success', 'Performance goal deleted successfully.');
    }
}