<?php

namespace App\Http\Controllers;

use App\Models\Organization;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use App\Traits\SendsOrganizationMail;

class OrganizationController extends Controller
{
    use SendsOrganizationMail;
    public function __construct()
    {
        $this->middleware('system_admin');
    }

    /**
     * Display a listing of organizations
     */
    public function index()
    {
        $organizations = Organization::withCount(['users', 'employees'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('systemadmin.organizations.index', compact('organizations'));
    }

    /**
     * Show the form for creating a new organization
     */
    public function create()
    {
        return view('systemadmin.organizations.create');
    }

    /**
     * Store a newly created organization
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:organizations',
            'admin_name' => 'required|string|max:255',
            'admin_email' => 'required|email|unique:users,email',
            'admin_password' => 'required|string|min:8|confirmed',
            'status' => 'required|in:active,inactive,suspended',
        ]);

        // Store plain password for email
        $plainPassword = $request->admin_password;

        // Create organization
        $organization = Organization::create([
            'name' => $request->name,
            'slug' => $request->slug ?: Str::slug($request->name),
            'status' => $request->status,
            'onboarding_step' => 'pending_verification',
            'settings' => [],
        ]);

        // Create organization admin user
        $adminUser = User::create([
            'name' => $request->admin_name,
            'email' => $request->admin_email,
            'password' => Hash::make($plainPassword),
            'organization_id' => $organization->id,
            'is_system_admin' => false,
            'must_verify_email' => true,
            'first_login_completed' => false,
            'initial_password' => $plainPassword, // Store for reference
        ]);

        // Assign organization admin role
        $adminUser->assignRole('org_admin');

        // Send welcome email with credentials using system email settings (not organization)
        // This is a system admin action, so we use .env email configuration
        $emailSent = $this->sendSystemMail(
            $adminUser->email,
            new \App\Mail\OrganizationAdminCreated($organization, $adminUser, $plainPassword)
        );

        if ($emailSent) {
            return redirect()->route('systemadmin.organizations.index')
                ->with('success', 'Organization created successfully! Welcome email sent to ' . $adminUser->email);
        } else {
            return redirect()->route('systemadmin.organizations.index')
                ->with('warning', 'Organization created successfully, but failed to send welcome email. Please check email configuration and logs.');
        }
    }

    /**
     * Display the specified organization
     */
    public function show(Organization $organization)
    {
        $organization->load(['users.roles', 'employees', 'departments']);
        
        $stats = [
            'total_users' => $organization->users()->count(),
            'total_employees' => $organization->employees()->count(),
            'total_departments' => $organization->departments()->count(),
            'admin_users' => $organization->users()->role('org_admin')->count(),
        ];

        return view('systemadmin.organizations.show', compact('organization', 'stats'));
    }

    /**
     * Show the form for editing the specified organization
     */
    public function edit(Organization $organization)
    {
        return view('systemadmin.organizations.edit', compact('organization'));
    }

    /**
     * Update the specified organization
     */
    public function update(Request $request, Organization $organization)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => ['nullable', 'string', 'max:255', Rule::unique('organizations')->ignore($organization->id)],
            'status' => 'required|in:active,inactive,suspended',
        ]);

        $organization->update([
            'name' => $request->name,
            'slug' => $request->slug ?: Str::slug($request->name),
            'status' => $request->status,
        ]);

        return redirect()->route('systemadmin.organizations.show', $organization)
            ->with('success', 'Organization updated successfully.');
    }

    /**
     * Remove the specified organization
     */
    public function destroy(Organization $organization)
    {
        // Check if organization has users or employees
        if ($organization->users()->count() > 0 || $organization->employees()->count() > 0) {
            return redirect()->route('systemadmin.organizations.index')
                ->with('error', 'Cannot delete organization with existing users or employees.');
        }

        $organization->delete();

        return redirect()->route('systemadmin.organizations.index')
            ->with('success', 'Organization deleted successfully.');
    }
}