<?php

namespace App\Http\Controllers;

use App\Models\JobOffer;
use App\Models\Employee;
use App\Models\User;
use App\Mail\WelcomeNewEmployee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;

class OfferResponseController extends Controller
{
    public function show($token)
    {
        $offer = JobOffer::where('offer_token', $token)
            ->with(['application.jobPosting.organization', 'application.jobPosting.department'])
            ->firstOrFail();

        if ($offer->isExpired()) {
            return view('candidate.offer.expired', compact('offer'));
        }

        return view('candidate.offer.view', compact('offer'));
    }

    public function accept($token)
    {
        $offer = JobOffer::where('offer_token', $token)->firstOrFail();

        if ($offer->status !== 'sent') {
            return redirect()->route('offer.view', $token)
                ->with('info', 'This offer has already been responded to.');
        }

        if ($offer->isExpired()) {
            return view('candidate.offer.expired', compact('offer'));
        }

        return view('candidate.offer.accept', compact('offer'));
    }

    public function storeAcceptance(Request $request, $token)
    {
        $offer = JobOffer::where('offer_token', $token)->firstOrFail();

        if ($offer->status !== 'sent') {
            return redirect()->route('offer.view', $token)
                ->with('error', 'This offer has already been responded to.');
        }

        $validated = $request->validate([
            'password' => 'required|string|min:8|confirmed',
            'accept_terms' => 'required|accepted',
        ]);

        DB::beginTransaction();
        try {
            // Create employee record
            $employee = Employee::create([
                'first_name' => $offer->application->first_name,
                'last_name' => $offer->application->last_name,
                'email' => $offer->application->email,
                'phone' => $offer->application->phone,
                'department_id' => $offer->application->jobPosting->department_id,
                'position_id' => $offer->application->jobPosting->position_id,
                'hire_date' => $offer->start_date,
                'employment_type' => $offer->application->jobPosting->employment_type,
                'salary' => $offer->salary,
                'status' => 'pending',
                'organization_id' => $offer->organization_id,
            ]);

            // Create user account
            $user = User::create([
                'name' => $offer->application->full_name,
                'email' => $offer->application->email,
                'password' => Hash::make($validated['password']),
                'organization_id' => $offer->organization_id,
            ]);

            // Assign employee role
            $user->assignRole('employee');

            // Update offer
            $offer->update([
                'status' => 'accepted',
                'responded_at' => now(),
            ]);

            // Update application
            $offer->application->update([
                'employee_id' => $employee->id,
                'converted_to_employee_at' => now(),
            ]);

            DB::commit();

            // Send welcome email
            Mail::to($user->email)->send(new WelcomeNewEmployee($user, $employee));

            // Log the user in
            auth()->login($user);

            return redirect()->route('dashboard')
                ->with('success', 'Welcome aboard! Your account has been created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Offer acceptance failed: ' . $e->getMessage());
            return back()->with('error', 'Something went wrong. Please try again.');
        }
    }

    public function decline($token)
    {
        $offer = JobOffer::where('offer_token', $token)->firstOrFail();

        if ($offer->status !== 'sent') {
            return redirect()->route('offer.view', $token)
                ->with('info', 'This offer has already been responded to.');
        }

        return view('candidate.offer.decline', compact('offer'));
    }

    public function storeDecline(Request $request, $token)
    {
        $offer = JobOffer::where('offer_token', $token)->firstOrFail();

        if ($offer->status !== 'sent') {
            return redirect()->route('offer.view', $token)
                ->with('error', 'This offer has already been responded to.');
        }

        $offer->update([
            'status' => 'declined',
            'responded_at' => now(),
            'candidate_notes' => $request->reason,
        ]);

        $offer->application->update(['status' => 'rejected']);

        return view('candidate.offer.declined', compact('offer'));
    }
}
