<?php

namespace App\Http\Controllers;

use App\Models\Application;
use App\Models\JobOffer;
use App\Mail\JobOfferSent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Barryvdh\DomPDF\Facade\Pdf;

class JobOfferController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_applications')->only(['show']);
        $this->middleware('permission:create_job_postings')->only(['create', 'store']);
        $this->middleware('permission:edit_job_postings')->only(['edit', 'update', 'send']);
    }

    public function create(Application $application)
    {
        if ($application->jobOffer) {
            return redirect()->route('job-offers.show', $application->jobOffer)
                ->with('info', 'An offer already exists for this application.');
        }

        return view('recruitment.offers.create', compact('application'));
    }

    public function store(Request $request, Application $application)
    {
        $validated = $request->validate([
            'position_title' => 'required|string|max:255',
            'salary' => 'required|numeric|min:0',
            'salary_period' => 'required|in:hourly,monthly,annually',
            'start_date' => 'required|date|after:today',
            'benefits' => 'nullable|string',
            'terms_and_conditions' => 'required|string',
        ]);

        $offer = JobOffer::create([
            'application_id' => $application->id,
            'position_title' => $validated['position_title'],
            'salary' => $validated['salary'],
            'salary_period' => $validated['salary_period'],
            'start_date' => $validated['start_date'],
            'benefits' => $validated['benefits'],
            'terms_and_conditions' => $validated['terms_and_conditions'],
            'status' => 'draft',
            'created_by' => auth()->id(),
            'organization_id' => auth()->user()->organization_id,
        ]);

        return redirect()->route('job-offers.show', $offer)
            ->with('success', 'Job offer created successfully!');
    }

    public function show(JobOffer $jobOffer)
    {
        // Check if user can access this offer (tenant scoping)
        if (!auth()->check() || $jobOffer->organization_id !== auth()->user()->organization_id) {
            abort(403, 'Access denied.');
        }

        $jobOffer->load(['application.jobPosting', 'creator']);
        return view('recruitment.offers.show', compact('jobOffer'));
    }

    public function edit(JobOffer $jobOffer)
    {
        // Check if user can access this offer (tenant scoping)
        if (!auth()->check() || $jobOffer->organization_id !== auth()->user()->organization_id) {
            abort(403, 'Access denied.');
        }

        if ($jobOffer->status !== 'draft') {
            return redirect()->route('job-offers.show', $jobOffer)
                ->with('error', 'Cannot edit an offer that has been sent.');
        }

        return view('recruitment.offers.edit', compact('jobOffer'));
    }

    public function update(Request $request, JobOffer $jobOffer)
    {
        // Check if user can access this offer (tenant scoping)
        if (!auth()->check() || $jobOffer->organization_id !== auth()->user()->organization_id) {
            abort(403, 'Access denied.');
        }

        if ($jobOffer->status !== 'draft') {
            return redirect()->route('job-offers.show', $jobOffer)
                ->with('error', 'Cannot edit an offer that has been sent.');
        }

        $validated = $request->validate([
            'position_title' => 'required|string|max:255',
            'salary' => 'required|numeric|min:0',
            'salary_period' => 'required|in:hourly,monthly,annually',
            'start_date' => 'required|date|after:today',
            'benefits' => 'nullable|string',
            'terms_and_conditions' => 'required|string',
        ]);

        $jobOffer->update($validated);

        return redirect()->route('job-offers.show', $jobOffer)
            ->with('success', 'Job offer updated successfully!');
    }

    public function send(JobOffer $jobOffer)
    {
        // Check if user can access this offer (tenant scoping)
        if (!auth()->check() || $jobOffer->organization_id !== auth()->user()->organization_id) {
            abort(403, 'Access denied.');
        }

        if ($jobOffer->status !== 'draft') {
            return redirect()->route('job-offers.show', $jobOffer)
                ->with('error', 'This offer has already been sent.');
        }

        // Generate PDF
        $pdf = Pdf::loadView('recruitment.offers.pdf', ['offer' => $jobOffer]);
        $filename = 'offer_' . $jobOffer->id . '.pdf';
        $path = 'offers/' . $jobOffer->organization_id . '/' . $filename;
        \Storage::disk('public')->put($path, $pdf->output());

        $jobOffer->update([
            'offer_letter_path' => $path,
            'status' => 'sent',
            'sent_at' => now(),
            'expires_at' => now()->addDays(7),
        ]);

        // Update application status
        $jobOffer->application->update(['status' => 'offered']);

        // Send email
        Mail::to($jobOffer->application->email)->send(new JobOfferSent($jobOffer));

        return redirect()->route('job-offers.show', $jobOffer)
            ->with('success', 'Job offer sent successfully!');
    }
}
