<?php

namespace App\Http\Controllers;

use App\Models\Application;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CandidatePortalController extends Controller
{
    /**
     * Show candidate tracking dashboard
     */
    public function show($token)
    {
        $application = Application::where('tracking_token', $token)
            ->with([
                'jobPosting:id,title,location,employment_type,organization_id,department_id',
                'jobPosting.department:id,name',
                'jobPosting.organization:id,name',
                'statusHistory.changedBy:id,name',
                'interviews.interviewer:id,name',
                'jobOffer'
            ])
            ->firstOrFail();

        return view('candidate.portal.dashboard', compact('application'));
    }

    /**
     * Show application timeline
     */
    public function timeline($token)
    {
        $application = Application::where('tracking_token', $token)
            ->with(['statusHistory.changedBy:id,name'])
            ->firstOrFail();

        return view('candidate.portal.timeline', compact('application'));
    }

    /**
     * Download CV
     */
    public function downloadCv($token)
    {
        $application = Application::where('tracking_token', $token)->firstOrFail();

        if (!$application->cv_path) {
            abort(404, 'CV file not found.');
        }

        $filePath = str_replace('public/', '', $application->cv_path);
        $fullPath = storage_path('app/public/' . $filePath);

        if (!file_exists($fullPath)) {
            abort(404, 'CV file not found on disk.');
        }

        $fileName = $application->full_name . '_CV.' . pathinfo($fullPath, PATHINFO_EXTENSION);

        return response()->download($fullPath, $fileName);
    }

    /**
     * Show update form (if allowed)
     */
    public function edit($token)
    {
        $application = Application::where('tracking_token', $token)
            ->with('jobPosting')
            ->firstOrFail();

        if (!$application->canBeUpdated()) {
            return redirect()->route('candidate.track', $token)
                ->with('error', 'This application can no longer be updated.');
        }

        return view('candidate.portal.edit', compact('application'));
    }

    /**
     * Update application
     */
    public function update(Request $request, $token)
    {
        $application = Application::where('tracking_token', $token)->firstOrFail();

        if (!$application->canBeUpdated()) {
            return redirect()->route('candidate.track', $token)
                ->with('error', 'This application can no longer be updated.');
        }

        $validated = $request->validate([
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'phone' => 'required|string|max:20',
            'cover_letter' => 'nullable|string|max:2000',
            'cv' => 'nullable|file|mimes:pdf,doc,docx|max:5120',
        ]);

        $updateData = [
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'phone' => $validated['phone'],
            'cover_letter' => $validated['cover_letter'],
            'updated_count' => $application->updated_count + 1,
            'last_updated_at' => now(),
        ];

        // Handle CV update
        if ($request->hasFile('cv')) {
            // Delete old CV
            if ($application->cv_path) {
                $oldPath = storage_path('app/public/' . str_replace('public/', '', $application->cv_path));
                if (file_exists($oldPath)) {
                    unlink($oldPath);
                }
            }

            // Store new CV
            $file = $request->file('cv');
            $filename = time() . '_' . $file->getClientOriginalName();
            $path = "applications/{$application->organization_id}/{$application->job_posting_id}";
            $storedPath = $file->storeAs($path, $filename, 'public');
            $updateData['cv_path'] = $storedPath;
        }

        $application->update($updateData);

        return redirect()->route('candidate.track', $token)
            ->with('success', 'Your application has been updated successfully!');
    }

    /**
     * Withdraw application
     */
    public function withdraw($token)
    {
        $application = Application::where('tracking_token', $token)->firstOrFail();

        if (!$application->canBeWithdrawn()) {
            return redirect()->route('candidate.track', $token)
                ->with('error', 'This application can no longer be withdrawn.');
        }

        return view('candidate.portal.withdraw', compact('application'));
    }

    /**
     * Confirm withdrawal
     */
    public function confirmWithdraw(Request $request, $token)
    {
        $application = Application::where('tracking_token', $token)->firstOrFail();

        if (!$application->canBeWithdrawn()) {
            return redirect()->route('candidate.track', $token)
                ->with('error', 'This application can no longer be withdrawn.');
        }

        $application->update([
            'status' => 'withdrawn',
            'notes' => ($application->notes ? $application->notes . "\n\n" : '') . 
                      'Withdrawn by candidate on ' . now()->format('Y-m-d H:i:s') .
                      ($request->reason ? "\nReason: " . $request->reason : '')
        ]);

        return view('candidate.portal.withdrawn', compact('application'));
    }
}
