<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class ForgotPasswordController extends Controller
{
    /**
     * Show the form for requesting a password reset link.
     */
    public function showLinkRequestForm()
    {
        return view('auth.passwords.email');
    }

    /**
     * Send a reset link to the given user.
     */
    public function sendResetLinkEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            throw ValidationException::withMessages($validator->errors()->toArray());
        }

        // Rate limiting: 5 attempts per hour
        $key = 'password-reset-' . $request->ip();
        if (cache()->has($key) && cache()->get($key) >= 5) {
            throw ValidationException::withMessages([
                'email' => ['Too many password reset attempts. Please try again later.']
            ]);
        }

        $status = Password::sendResetLink(
            $request->only('email')
        );

        // Increment attempt counter
        $attempts = cache()->get($key, 0) + 1;
        cache()->put($key, $attempts, now()->addHour());

        if ($status === Password::RESET_LINK_SENT) {
            return back()->with('status', __($status));
        }

        throw ValidationException::withMessages([
            'email' => [__($status)],
        ]);
    }
}