<?php

namespace App\Http\Controllers;

use App\Models\Application;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Response;

class ApplicationController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_applications');
    }

    /**
     * Download CV file
     */
    public function downloadCv(Application $application)
    {
        // Check if user can access this application (tenant scoping)
        if (!$application->belongsToCurrentOrganization()) {
            \Log::warning('CV access denied for application ' . $application->id . ' by user ' . auth()->id());
            abort(403, 'Access denied.');
        }

        if (!$application->cv_path) {
            abort(404, 'CV file not found.');
        }

        // Clean the path - remove 'public/' prefix if it exists
        $filePath = str_replace('public/', '', $application->cv_path);
        $fullPath = storage_path('app/public/' . $filePath);

        \Log::info('Attempting to download CV', [
            'application_id' => $application->id,
            'cv_path' => $application->cv_path,
            'file_path' => $filePath,
            'full_path' => $fullPath,
            'file_exists' => file_exists($fullPath)
        ]);

        // Check if file exists
        if (!file_exists($fullPath)) {
            \Log::error('CV file not found on disk: ' . $fullPath);
            abort(404, 'CV file not found on disk.');
        }

        // Get file info
        $fileName = $application->full_name . '_CV_' . $application->applied_at->format('Y-m-d') . '.' . pathinfo($fullPath, PATHINFO_EXTENSION);

        // Return file download response
        return Response::download($fullPath, $fileName);
    }

    /**
     * View CV in browser
     */
    public function viewCv(Application $application)
    {
        // Check if user can access this application (tenant scoping)
        if (!$application->belongsToCurrentOrganization()) {
            abort(403, 'Access denied.');
        }

        if (!$application->cv_path) {
            abort(404, 'CV file not found.');
        }

        // Clean the path - remove 'public/' prefix if it exists
        $filePath = str_replace('public/', '', $application->cv_path);
        $fullPath = storage_path('app/public/' . $filePath);

        // Check if file exists
        if (!file_exists($fullPath)) {
            abort(404, 'CV file not found on disk.');
        }

        // Get MIME type
        $mimeType = mime_content_type($fullPath);
        
        // Return file for viewing in browser
        return Response::file($fullPath, [
            'Content-Type' => $mimeType,
            'Content-Disposition' => 'inline; filename="' . basename($fullPath) . '"'
        ]);
    }

    /**
     * Update application status
     */
    public function updateStatus(Request $request, Application $application)
    {
        // Check authentication and permissions
        if (!auth()->check()) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        // Check if user can access this application (tenant scoping)
        if (!$application->belongsToCurrentOrganization()) {
            return response()->json(['success' => false, 'message' => 'Access denied'], 403);
        }

        $validated = $request->validate([
            'status' => 'required|in:screening,shortlisted,interviewed,offered,rejected,withdrawn'
        ]);

        $oldStatus = $application->status;
        $newStatus = $validated['status'];

        // Update application status
        $application->update([
            'status' => $newStatus
        ]);

        // Create status history record if the relationship exists
        if (method_exists($application, 'statusHistory')) {
            try {
                $application->statusHistory()->create([
                    'from_status' => $oldStatus,
                    'to_status' => $newStatus,
                    'changed_by' => auth()->id(),
                    'notes' => 'Status updated via job posting management',
                ]);
            } catch (\Exception $e) {
                \Log::warning('Failed to create status history: ' . $e->getMessage());
            }
        }

        // Send email notification to candidate
        try {
            if (class_exists('\App\Mail\ApplicationStatusChanged')) {
                \Mail::to($application->email)->send(
                    new \App\Mail\ApplicationStatusChanged($application, $oldStatus, $newStatus)
                );
            }
        } catch (\Exception $e) {
            \Log::error('Failed to send status update email: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Application status updated successfully',
            'old_status' => $oldStatus,
            'new_status' => $newStatus
        ]);
    }
}
