<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\MpesaService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class MpesaCallbackController extends Controller
{
    protected $mpesaService;

    public function __construct(MpesaService $mpesaService)
    {
        $this->mpesaService = $mpesaService;
    }

    /**
     * Handle M-Pesa STK Push callback
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function callback(Request $request): JsonResponse
    {
        try {
            // Log the raw callback
            Log::info('M-Pesa callback received', [
                'headers' => $request->headers->all(),
                'body' => $request->all(),
            ]);

            // Process the callback
            $success = $this->mpesaService->processCallback($request->all());

            // M-Pesa expects a specific response format
            return response()->json([
                'ResultCode' => $success ? 0 : 1,
                'ResultDesc' => $success ? 'Success' : 'Failed',
            ]);

        } catch (\Exception $e) {
            Log::error('Error in M-Pesa callback handler', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'ResultCode' => 1,
                'ResultDesc' => 'Internal server error',
            ], 500);
        }
    }

    /**
     * Handle M-Pesa C2B validation
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function validation(Request $request): JsonResponse
    {
        Log::info('M-Pesa C2B validation request', $request->all());

        // Accept all transactions
        return response()->json([
            'ResultCode' => 0,
            'ResultDesc' => 'Accepted',
        ]);
    }

    /**
     * Handle M-Pesa C2B confirmation
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function confirmation(Request $request): JsonResponse
    {
        Log::info('M-Pesa C2B confirmation', $request->all());

        // Process C2B payment if needed
        // For now, just acknowledge receipt

        return response()->json([
            'ResultCode' => 0,
            'ResultDesc' => 'Success',
        ]);
    }

    /**
     * Test callback endpoint (for development)
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function test(Request $request): JsonResponse
    {
        if (!config('app.debug')) {
            abort(404);
        }

        // Simulate a successful M-Pesa callback
        $testCallback = [
            'Body' => [
                'stkCallback' => [
                    'MerchantRequestID' => '29115-34620561-1',
                    'CheckoutRequestID' => 'ws_CO_191220191020363925',
                    'ResultCode' => 0,
                    'ResultDesc' => 'The service request is processed successfully.',
                    'CallbackMetadata' => [
                        'Item' => [
                            ['Name' => 'Amount', 'Value' => 1.00],
                            ['Name' => 'MpesaReceiptNumber', 'Value' => 'NLJ7RT61SV'],
                            ['Name' => 'TransactionDate', 'Value' => 20191219102115],
                            ['Name' => 'PhoneNumber', 'Value' => 254708374149],
                        ],
                    ],
                ],
            ],
        ];

        $success = $this->mpesaService->processCallback($testCallback);

        return response()->json([
            'success' => $success,
            'message' => $success ? 'Test callback processed successfully' : 'Test callback failed',
        ]);
    }
}
