<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Services\SecurityService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    protected $securityService;

    public function __construct(SecurityService $securityService)
    {
        $this->securityService = $securityService;
    }

    /**
     * API Login
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
            'device_name' => 'required|string|max:255'
        ]);

        // Check rate limiting
        if ($this->securityService->isLoginRateLimited($request->ip())) {
            throw ValidationException::withMessages([
                'email' => ['Too many login attempts. Please try again later.']
            ]);
        }

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            $this->securityService->incrementLoginAttempts($request->ip());
            
            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.']
            ]);
        }

        // Check if account is locked
        if ($this->securityService->isAccountLocked($user)) {
            throw ValidationException::withMessages([
                'email' => ['Account is temporarily locked. Please try again later.']
            ]);
        }

        // Check email verification
        if (!$user->hasVerifiedEmail()) {
            throw ValidationException::withMessages([
                'email' => ['Please verify your email address before accessing the API.']
            ]);
        }

        // Clear failed attempts
        $this->securityService->clearLoginAttempts($request->ip());

        // Create token with abilities based on user role
        $abilities = $this->getUserAbilities($user);
        $token = $user->createToken($request->device_name, $abilities);

        // Update login information
        $user->update([
            'last_login_at' => now(),
            'login_ip' => $request->ip(),
            'login_user_agent' => $request->userAgent()
        ]);

        // Log successful API login
        activity()
            ->performedOn($user)
            ->withProperties([
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'device_name' => $request->device_name,
                'login_type' => 'api'
            ])
            ->log('API login successful');

        return response()->json([
            'user' => $user->load('roles', 'permissions'),
            'token' => $token->plainTextToken,
            'abilities' => $abilities
        ]);
    }

    /**
     * API Logout
     */
    public function logout(Request $request)
    {
        // Revoke current token
        $request->user()->currentAccessToken()->delete();

        activity()
            ->performedOn($request->user())
            ->withProperties(['login_type' => 'api'])
            ->log('API logout');

        return response()->json(['message' => 'Logged out successfully']);
    }

    /**
     * Refresh token
     */
    public function refresh(Request $request)
    {
        $request->validate([
            'device_name' => 'required|string|max:255'
        ]);

        $user = $request->user();
        
        // Revoke current token
        $user->currentAccessToken()->delete();
        
        // Create new token
        $abilities = $this->getUserAbilities($user);
        $token = $user->createToken($request->device_name, $abilities);

        activity()
            ->performedOn($user)
            ->withProperties(['login_type' => 'api'])
            ->log('API token refreshed');

        return response()->json([
            'token' => $token->plainTextToken,
            'abilities' => $abilities
        ]);
    }

    /**
     * Get user abilities based on roles and permissions
     */
    protected function getUserAbilities(User $user): array
    {
        $abilities = ['read'];

        if ($user->isSystemAdmin()) {
            return ['*']; // All abilities for system admin
        }

        // Add abilities based on permissions
        if ($user->can('create_employees')) $abilities[] = 'employees:create';
        if ($user->can('edit_employees')) $abilities[] = 'employees:update';
        if ($user->can('delete_employees')) $abilities[] = 'employees:delete';
        if ($user->can('view_payroll')) $abilities[] = 'payroll:read';
        if ($user->can('create_payroll')) $abilities[] = 'payroll:create';
        if ($user->can('edit_payroll')) $abilities[] = 'payroll:update';

        return array_unique($abilities);
    }
}