<?php

namespace App\Http\Controllers;

use App\Models\Announcement;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

class AnnouncementController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorize('viewAny', Announcement::class);

        $query = Announcement::with('creator');

        // If user doesn't have manage permission, only show published announcements
        if (!auth()->user()->hasPermissionTo('manage_announcements')) {
            $query->where('status', 'published')
                  ->where('active', true)
                  ->where('start_date', '<=', now())
                  ->where(function ($q) {
                      $q->whereNull('end_date')->orWhere('end_date', '>=', now());
                  });
        }

        $query->orderBy('is_pinned', 'desc')
              ->orderBy('created_at', 'desc');

        // Apply filters
        if ($request->filled('status') && auth()->user()->hasPermissionTo('manage_announcements')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('priority')) {
            $query->where('priority', $request->priority);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('content', 'like', "%{$search}%")
                  ->orWhere('excerpt', 'like', "%{$search}%");
            });
        }

        $announcements = $query->paginate(15);

        // Statistics - only for users with manage permission
        $stats = [];
        if (auth()->user()->hasPermissionTo('manage_announcements')) {
            $stats = [
                'total' => Announcement::count(),
                'published' => Announcement::where('status', 'published')->count(),
                'draft' => Announcement::where('status', 'draft')->count(),
                'archived' => Announcement::where('status', 'archived')->count(),
                'pinned' => Announcement::where('is_pinned', true)->count(),
                'urgent' => Announcement::where('priority', 'urgent')->count(),
            ];
        } else {
            $stats = [
                'total' => Announcement::where('status', 'published')->where('active', true)->count(),
                'published' => Announcement::where('status', 'published')->where('active', true)->count(),
                'draft' => 0,
                'archived' => 0,
                'pinned' => Announcement::where('is_pinned', true)->where('status', 'published')->where('active', true)->count(),
                'urgent' => Announcement::where('priority', 'urgent')->where('status', 'published')->where('active', true)->count(),
            ];
        }

        return view('announcements.index', compact('announcements', 'stats'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorize('create', Announcement::class);

        $users = User::select('id', 'name')->orderBy('name')->get();
        
        return view('announcements.create', compact('users'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorize('create', Announcement::class);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'excerpt' => 'nullable|string|max:500',
            'priority' => 'required|in:low,medium,high,urgent',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'is_pinned' => 'boolean',
            'active' => 'boolean',
            'status' => 'required|in:draft,published',
        ]);

        // Generate unique slug
        $slug = Str::slug($validated['title']);
        $originalSlug = $slug;
        $counter = 1;
        
        while (Announcement::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        $announcement = Announcement::create([
            'title' => $validated['title'],
            'slug' => $slug,
            'content' => $validated['content'],
            'excerpt' => $validated['excerpt'],
            'created_by' => Auth::id(),
            'priority' => $validated['priority'],
            'start_date' => $validated['start_date'],
            'end_date' => $validated['end_date'],
            'is_pinned' => $request->boolean('is_pinned'),
            'active' => $request->boolean('active', true),
            'status' => $validated['status'],
        ]);

        return redirect()->route('announcements.show', $announcement)
            ->with('success', 'Announcement created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Announcement $announcement)
    {
        $this->authorize('view', $announcement);

        $announcement->load('creator');

        return view('announcements.show', compact('announcement'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Announcement $announcement)
    {
        $this->authorize('update', $announcement);

        $users = User::select('id', 'name')->orderBy('name')->get();

        return view('announcements.edit', compact('announcement', 'users'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Announcement $announcement)
    {
        $this->authorize('update', $announcement);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'excerpt' => 'nullable|string|max:500',
            'priority' => 'required|in:low,medium,high,urgent',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'is_pinned' => 'boolean',
            'active' => 'boolean',
            'status' => 'required|in:draft,published,archived',
        ]);

        // Update slug if title changed
        if ($announcement->title !== $validated['title']) {
            $slug = Str::slug($validated['title']);
            $originalSlug = $slug;
            $counter = 1;
            
            while (Announcement::where('slug', $slug)->where('id', '!=', $announcement->id)->exists()) {
                $slug = $originalSlug . '-' . $counter;
                $counter++;
            }
            $announcement->slug = $slug;
        }

        $announcement->update([
            'title' => $validated['title'],
            'slug' => $announcement->slug,
            'content' => $validated['content'],
            'excerpt' => $validated['excerpt'],
            'priority' => $validated['priority'],
            'start_date' => $validated['start_date'],
            'end_date' => $validated['end_date'],
            'is_pinned' => $request->boolean('is_pinned'),
            'active' => $request->boolean('active'),
            'status' => $validated['status'],
        ]);

        return redirect()->route('announcements.show', $announcement)
            ->with('success', 'Announcement updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Announcement $announcement)
    {
        $this->authorize('delete', $announcement);

        $announcement->delete();

        return redirect()->route('announcements.index')
            ->with('success', 'Announcement deleted successfully.');
    }

    /**
     * Toggle pin status
     */
    public function togglePin(Announcement $announcement)
    {
        $this->authorize('update', $announcement);

        $announcement->update([
            'is_pinned' => !$announcement->is_pinned
        ]);

        return back()->with('success', 'Announcement pin status updated.');
    }

    /**
     * Publish announcement
     */
    public function publish(Announcement $announcement)
    {
        $this->authorize('update', $announcement);

        $announcement->update(['status' => 'published']);

        return back()->with('success', 'Announcement published successfully.');
    }

    /**
     * Archive announcement
     */
    public function archive(Announcement $announcement)
    {
        $this->authorize('update', $announcement);

        $announcement->update(['status' => 'archived']);

        return back()->with('success', 'Announcement archived successfully.');
    }
}