<?php

namespace App\Console\Commands;

use App\Mail\ApplicationReceived;
use App\Mail\ApplicationStatusChanged;
use App\Mail\InterviewScheduled;
use App\Mail\JobOfferSent;
use App\Models\Application;
use App\Models\Interview;
use App\Models\JobOffer;
use App\Models\JobPosting;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;

class TestRecruitmentEmails extends Command
{
    protected $signature = 'recruitment:test-emails {email} {--application-id=} {--job-posting-id=}';
    protected $description = 'Test recruitment email flow by sending sample emails';

    public function handle()
    {
        $testEmail = $this->argument('email');
        $applicationId = $this->option('application-id');
        $jobPostingId = $this->option('job-posting-id');

        $this->info("Testing recruitment emails to: {$testEmail}");
        $this->newLine();

        try {
            // Test 1: Application Received Email
            $this->info('1. Testing Application Received Email...');
            $application = $applicationId 
                ? Application::with('jobPosting.organization')->findOrFail($applicationId)
                : Application::with('jobPosting.organization')->latest()->first();

            if (!$application) {
                $this->error('No applications found. Please create an application first.');
                return 1;
            }

            Mail::to($testEmail)->send(new ApplicationReceived($application, $application->jobPosting));
            $this->info('✓ Application Received email sent');
            $this->newLine();

            // Test 2: Application Status Changed Email
            $this->info('2. Testing Application Status Changed Email...');
            Mail::to($testEmail)->send(
                new ApplicationStatusChanged($application, 'new', 'shortlisted', null)
            );
            $this->info('✓ Application Status Changed email sent');
            $this->newLine();

            // Test 3: Interview Scheduled Email (if interview exists)
            $interview = Interview::with('application.jobPosting.organization')->where('application_id', $application->id)->first();
            if ($interview) {
                $this->info('3. Testing Interview Scheduled Email...');
                Mail::to($testEmail)->send(new InterviewScheduled($interview, $application));
                $this->info('✓ Interview Scheduled email sent');
                $this->newLine();
            } else {
                $this->warn('3. Skipping Interview Scheduled Email (no interview found)');
                $this->newLine();
            }

            // Test 4: Job Offer Email (if offer exists)
            $offer = JobOffer::with('application.jobPosting.organization')->where('application_id', $application->id)->first();
            if ($offer) {
                $this->info('4. Testing Job Offer Email...');
                Mail::to($testEmail)->send(new JobOfferSent($offer));
                $this->info('✓ Job Offer email sent');
                $this->newLine();
            } else {
                $this->warn('4. Skipping Job Offer Email (no offer found)');
                $this->newLine();
            }

            $this->info('✅ All available recruitment emails have been sent!');
            $this->info("Check {$testEmail} for the test emails.");
            
            // Show organization email settings
            $organization = $application->jobPosting->organization;
            $this->newLine();
            $this->info('Organization Email Settings:');
            $this->table(
                ['Setting', 'Value'],
                [
                    ['Use Custom Mail', $organization->use_custom_mail ? 'Yes' : 'No'],
                    ['Mail Host', $organization->mail_host ?? 'N/A'],
                    ['Mail Port', $organization->mail_port ?? 'N/A'],
                    ['From Address', $organization->mail_from_address ?? 'N/A'],
                    ['From Name', $organization->mail_from_name ?? 'N/A'],
                ]
            );

            return 0;
        } catch (\Exception $e) {
            $this->error('Error sending test emails: ' . $e->getMessage());
            $this->error($e->getTraceAsString());
            return 1;
        }
    }
}
