<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Organization;
use App\Models\User;
use App\Models\Employee;
use App\Models\Department;
use Illuminate\Support\Facades\Hash;

class TestMultitenancy extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:multitenancy';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test multitenancy functionality';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🧪 Testing Multitenancy Setup...');
        
        // Test 1: Check if SytemAdmin exists
        $systemAdmin = User::where('is_system_admin', true)->first();
        if ($systemAdmin) {
            $this->info("✅ SytemAdmin exists: {$systemAdmin->email}");
        } else {
            $this->error("❌ SytemAdmin not found");
            return;
        }
        
        // Test 2: Check if Qaya Creatives exists
        $defaultOrg = Organization::where('slug', 'default')->first();
        if ($defaultOrg) {
            $this->info("✅ Qaya Creatives exists: {$defaultOrg->name}");
        } else {
            $this->error("❌ Qaya Creatives not found");
            return;
        }
        
        // Test 3: Check if regular users belong to organization
        $regularUsers = User::where('is_system_admin', false)->whereNotNull('organization_id')->count();
        $this->info("✅ Regular users with organization: {$regularUsers}");
        
        // Test 4: Check if employees belong to organization
        $employees = Employee::whereNotNull('organization_id')->count();
        $this->info("✅ Employees with organization: {$employees}");
        
        // Test 5: Check if departments belong to organization
        $departments = Department::whereNotNull('organization_id')->count();
        $this->info("✅ Departments with organization: {$departments}");
        
        // Test 6: Create a test organization
        $testOrg = Organization::create([
            'name' => 'Test Organization',
            'slug' => 'test-org',
            'status' => 'active',
        ]);
        
        $testUser = User::create([
            'name' => 'Test Org Admin',
            'email' => 'test-org-qayacreatives@gmail.com',
            'password' => Hash::make('password'),
            'organization_id' => $testOrg->id,
        ]);
        
        $testUser->assignRole('org_admin');
        
        $this->info("✅ Created test organization: {$testOrg->name}");
        $this->info("✅ Created test org admin: {$testUser->email}");
        
        // Test 7: Test tenant scoping
        app()->instance('tenant.organization_id', $testOrg->id);
        
        $testDept = Department::create([
            'name' => 'Test Department',
            'active' => true,
        ]);
        
        if ($testDept->organization_id === $testOrg->id) {
            $this->info("✅ Tenant scoping works - department auto-assigned to organization");
        } else {
            $this->error("❌ Tenant scoping failed");
        }
        
        // Clean up test data
        $testDept->delete();
        $testUser->delete();
        $testOrg->delete();
        
        $this->info("🧹 Cleaned up test data");
        $this->info("🎉 Multitenancy test completed successfully!");
        
        $this->newLine();
        $this->info("📋 Login Credentials:");
        $this->info("SystemAdmin: systemadmin@example.com / password");
        $this->info("Regular Admin: qayacreatives@gmail.com / password");
        $this->info("SystemAdmin URL: /systemadmin/login");
        $this->info("Regular URL: /login");
    }
}