<?php

namespace App\Console\Commands;

use App\Models\User;
use Illuminate\Console\Command;
use Spatie\OneTimePasswords\Models\OneTimePassword;

class GetOtpCode extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'otp:get {email : The user email address}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Get the current OTP code for a user (for testing purposes)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $email = $this->argument('email');
        
        $user = User::where('email', $email)->first();
        
        if (!$user) {
            $this->error("User not found: {$email}");
            return 1;
        }
        
        $otp = OneTimePassword::where('authenticatable_type', User::class)
            ->where('authenticatable_id', $user->id)
            ->where('expires_at', '>', now())
            ->latest()
            ->first();
        
        if (!$otp) {
            $this->warn("No active OTP found for: {$email}");
            $this->info("Generate one by visiting /email/verify or running:");
            $this->line("php artisan otp:generate {$email}");
            return 1;
        }
        
        $this->info("OTP Code for {$user->name} ({$email}):");
        $this->line('');
        $this->line("  Code: {$otp->password}");
        $this->line("  Expires: {$otp->expires_at->diffForHumans()}");
        $this->line('');
        
        if ($otp->isExpired()) {
            $this->warn('⚠️  This OTP has expired!');
        }
        
        return 0;
    }
}
