<?php

namespace App\Console\Commands;

use App\Models\AuditLog;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class AuditCleanupCommand extends Command
{
    protected $signature = 'audit:cleanup 
                           {--days= : Override default retention days}
                           {--category= : Clean up specific category only}
                           {--dry-run : Show what would be deleted without actually deleting}
                           {--force : Force cleanup without confirmation}';

    protected $description = 'Clean up old audit logs based on retention policies';

    public function handle(): int
    {
        $this->info('Starting audit log cleanup...');

        $dryRun = $this->option('dry-run');
        $force = $this->option('force');
        $categoryFilter = $this->option('category');
        $daysOverride = $this->option('days');

        // Get retention policies
        $retentionPolicies = config('audit.retention.by_category', []);
        $defaultRetention = $daysOverride ?: config('audit.retention.default_days', 365);

        if ($categoryFilter) {
            $retentionPolicies = [$categoryFilter => $retentionPolicies[$categoryFilter] ?? $defaultRetention];
        }

        $totalDeleted = 0;
        $totalSize = 0;

        foreach ($retentionPolicies as $category => $retentionDays) {
            if ($retentionDays <= 0) {
                $this->info("Skipping category '{$category}' - retention set to keep forever");
                continue;
            }

            $cutoffDate = now()->subDays($retentionDays);
            
            $query = AuditLog::where('category', $category)
                ->where('created_at', '<', $cutoffDate);

            $count = $query->count();
            
            if ($count === 0) {
                $this->info("No old records found for category '{$category}'");
                continue;
            }

            // Calculate approximate size
            $sampleRecord = $query->first();
            $approximateSize = $sampleRecord ? strlen(json_encode($sampleRecord->toArray())) * $count : 0;
            $totalSize += $approximateSize;

            $this->info("Category '{$category}': {$count} records older than {$retentionDays} days");

            if ($dryRun) {
                $this->warn("[DRY RUN] Would delete {$count} records from category '{$category}'");
                continue;
            }

            if (!$force && !$this->confirm("Delete {$count} records from category '{$category}'?")) {
                $this->info("Skipped category '{$category}'");
                continue;
            }

            // Delete in batches for better performance
            $batchSize = config('audit.performance.batch_size', 1000);
            $deleted = 0;

            $this->info("Deleting records in batches of {$batchSize}...");
            $progressBar = $this->output->createProgressBar($count);

            while (true) {
                $batch = AuditLog::where('category', $category)
                    ->where('created_at', '<', $cutoffDate)
                    ->limit($batchSize)
                    ->get();

                if ($batch->isEmpty()) {
                    break;
                }

                $batchIds = $batch->pluck('id')->toArray();
                $batchDeleted = AuditLog::whereIn('id', $batchIds)->delete();
                
                $deleted += $batchDeleted;
                $progressBar->advance($batchDeleted);

                // Small delay to prevent overwhelming the database
                usleep(10000); // 10ms
            }

            $progressBar->finish();
            $this->newLine();
            
            $totalDeleted += $deleted;
            $this->info("Deleted {$deleted} records from category '{$category}'");
        }

        // Clean up records with no category (fallback)
        if (!$categoryFilter) {
            $cutoffDate = now()->subDays($defaultRetention);
            $orphanQuery = AuditLog::whereNull('category')
                ->orWhere('category', '')
                ->where('created_at', '<', $cutoffDate);

            $orphanCount = $orphanQuery->count();
            
            if ($orphanCount > 0) {
                $this->info("Found {$orphanCount} records with no category");
                
                if (!$dryRun && ($force || $this->confirm("Delete {$orphanCount} uncategorized records?"))) {
                    $orphanDeleted = $orphanQuery->delete();
                    $totalDeleted += $orphanDeleted;
                    $this->info("Deleted {$orphanDeleted} uncategorized records");
                }
            }
        }

        // Optimize table after cleanup
        if (!$dryRun && $totalDeleted > 0) {
            $this->info('Optimizing audit_logs table...');
            DB::statement('OPTIMIZE TABLE audit_logs');
        }

        $sizeFormatted = $this->formatBytes($totalSize);
        
        if ($dryRun) {
            $this->info("DRY RUN COMPLETE: Would delete {$totalDeleted} records (~{$sizeFormatted})");
        } else {
            $this->info("CLEANUP COMPLETE: Deleted {$totalDeleted} records (~{$sizeFormatted})");
        }

        return 0;
    }

    private function formatBytes(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= pow(1024, $pow);
        
        return round($bytes, 2) . ' ' . $units[$pow];
    }
}